/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.guest.impl.converters;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Properties;
import javax.mail.Address;
import javax.mail.BodyPart;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeMessage;
import org.bouncycastle.openpgp.PGPPublicKey;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.common.util.FunctionalExceptionUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.impl.lookup.DefaultRecipKeyLookupStrategy;
import com.openexchange.guard.keymanagement.services.lookup.RecipKeyLookupStrategy;
import com.openexchange.guard.mime.helpfiles.HelpFileService;
import com.openexchange.guard.user.OXGuardUser;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.java.Streams;
import com.openexchange.pgp.core.PGPDecrypter;
import com.openexchange.pgp.core.PGPKeyRetrievalStrategy;
import com.openexchange.pgp.mail.PGPMimeService;

/**
 * {@link BlobToMimeConverter} converts a guest item which is stored as plain PGP Blob (&lt;= Guard2.8)  to a proper PGP/MIME object.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public class BlobToMimeConverter implements GuestItemConverter {

    private static final String PGP_MARKER = "-----BEGIN PGP MESSAGE-----";
    private static final int    PEEK_SIZE  = 30;

    private final PGPMimeService          pgpMimeService;
    private final PGPKeyRetrievalStrategy keyRetrievalStrategy;
    private final HelpFileService         helpFileService;
    private final GuardConfigurationService     guardConfigurationService;

    /**
     * Initializes a new {@link BlobToMimeConverter}.
     *
     * @param pgpMimeService The {@link PGPMimeServer} to use
     * @param keyRetrievalStrategy The strategy for fetching the guest's private key for decryption
     * @param helpFileService The service for dealing with help-files
     * @param guardConfigurationService The configuration Service to use
     */
    public BlobToMimeConverter(PGPMimeService pgpMimeService,
                               PGPKeyRetrievalStrategy keyRetrievalStrategy,
                               HelpFileService helpFileService,
                               GuardConfigurationService guardConfigurationService) {
        this.pgpMimeService = Objects.requireNonNull(pgpMimeService, "pgpMimeService must not be null");
        this.keyRetrievalStrategy = Objects.requireNonNull(keyRetrievalStrategy, "keyRetrievalStrategy must not be null");
        this.helpFileService = Objects.requireNonNull(helpFileService, "helpFileService must not be null");
        this.guardConfigurationService = Objects.requireNonNull(guardConfigurationService, "guardConfigurationService must not be null");
    }

    /**
     * Internal method to check whether or not the given stream is a raw, ASCII-armored PGP blob.
     *
     * @param inputStream The stream to check
     * @return true, if the guest item is an ASCII armored PGP blob, false otherwise
     * @throws IOException
     */
    private boolean isPGPBlob(InputStream inputStream) throws IOException {
        try {
            byte[] peekedData = new byte[PEEK_SIZE];
            inputStream.mark(PEEK_SIZE);
            int read = inputStream.read(peekedData);
            if (read > 0) {
                String stringData = new String(peekedData, StandardCharsets.UTF_8);
                return stringData.contains(PGP_MARKER);
            }
        } finally {
            inputStream.reset();
        }
        return false;
    }

    /**
     * Internal method to decrypt a raw PGP blob
     *
     * @param guestIdentity The identity containing the key for decryption
     * @param blobData The data to decrypt
     * @return An {@link InputStream} to the decrypted data
     * @throws Exception
     */
    private InputStream decryptBlob(UserIdentity guestIdentity, InputStream blobData) throws Exception {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        PGPDecrypter decrypter = new PGPDecrypter(this.keyRetrievalStrategy);
        decrypter.decrypt(blobData, output, guestIdentity.getIdentity(), guestIdentity.getPassword());
        return new ByteArrayInputStream(output.toByteArray());
    }

    private RecipKeyLookupStrategy getRecipKeyLookupForGuest (UserIdentity guestIdentity) throws NoSuchAlgorithmException, UnsupportedEncodingException {
        return new DefaultRecipKeyLookupStrategy(
            guestIdentity.getOXGuardUser().getId(),
            guestIdentity.getOXGuardUser().getContextId(),
            CipherUtil.getMD5(guestIdentity.getIdentity()));
    }
    /**
     * Internal method to get the public keys for a message's recipients
     *
     * @param guestIdentity the owner of the message
     * @param mimeMessage The mimeMessage containing other recipients
     * @return A list of public keys
     * @throws MessagingException
     * @throws OXException
     * @throws NoSuchAlgorithmException
     * @throws UnsupportedEncodingException
     */
    private List<PGPPublicKey> getRecipientKeysFrom(UserIdentity guestIdentity, MimeMessage mimeMessage) throws MessagingException, OXException, NoSuchAlgorithmException, UnsupportedEncodingException {
        Address[] allRecipients = mimeMessage.getAllRecipients();
        if (allRecipients.length > 0) {
            ArrayList<PGPPublicKey> ret = new ArrayList<PGPPublicKey>(allRecipients.length);
            OXGuardUser guest = guestIdentity.getOXGuardUser();
            if (guest != null && guest.isGuest()) {
                int remoteHkpTimeout = guardConfigurationService.getIntProperty(GuardProperty.remoteKeyLookupTimeout);
                RecipKeyLookupStrategy recipKeyLookupStrategy = getRecipKeyLookupForGuest(guestIdentity);
                for (Address address : allRecipients) {
                    RecipKey recipKey = recipKeyLookupStrategy.lookup(((InternetAddress) address).getAddress(), remoteHkpTimeout);
                    if (recipKey == null) {
                        throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_MAIL_ONLY_ERROR.create(address.toString());
                    }
                    PGPPublicKey encryptionKey = recipKey.getEncryptionKey();
                    if (encryptionKey != null) {
                        ret.add(encryptionKey);
                    } else {
                        throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_MAIL_ONLY_ERROR.create(address.toString());
                    }
                }
                return ret;
            } else {
                throw GuardCoreExceptionCodes.NOT_A_GUEST_ACCOUNT.create();
            }
        }
        return new ArrayList<PGPPublicKey>();
    }

    private Collection<BodyPart> getAdditionalHelpFile(UserIdentity guestIdentity) throws OXException, NoSuchAlgorithmException, UnsupportedEncodingException {
        if (guestIdentity.getOXUser() == null) {  // In upgrading old guest file, no user Identity for the OX User
            // See if we can pull their key with locale available
            RecipKeyLookupStrategy recipKeyLookupStrategy = getRecipKeyLookupForGuest(guestIdentity);
            int remoteHkpTimeout = guardConfigurationService.getIntProperty(GuardProperty.remoteKeyLookupTimeout);
            RecipKey recipKey = recipKeyLookupStrategy.lookup(guestIdentity.getIdentity(), remoteHkpTimeout);
            if (recipKey != null) {
                return helpFileService.getHelpFiles(recipKey.getUserid(), recipKey.getCid(), recipKey.getLocale());
            } else {
                return Collections.emptyList();
            }

        }
        return helpFileService.getHelpFiles(guestIdentity.getOXUser().getId(), guestIdentity.getOXUser().getContextId(), guestIdentity.getOXUser().getLocale());
    }

    /**
     * Internal method to attach a Help file to the given message
     *
     * @param guestIdentity The guest
     * @param message The message to attach the help file to
     * @throws IOException
     * @throws MessagingException
     * @throws OXException
     * @throws NoSuchAlgorithmException
     */
    private void addAdditionalHelpFiles(UserIdentity guestIdentity, MimeMessage message) throws OXException, IOException, MessagingException, NoSuchAlgorithmException {
        Multipart content = (Multipart) message.getContent();
        getAdditionalHelpFile(guestIdentity).forEach(FunctionalExceptionUtil.toExceptionThrowingConsumer(helpFile -> content.addBodyPart(helpFile)));
        message.setContent(content);
    }

    /**
     * Internal method to convert the given guest data into proper PGP/MIME data.
     *
     * @param guestIdentity The guest
     * @param guestEmailData The mail data to convert
     * @param output The {@link OutputStream} to write the converted data to
     * @throws OXException
     */
    private void convertToPGPMime(UserIdentity guestIdentity, InputStream guestEmailData, OutputStream output) throws OXException {
        try {
            //Decrypt then RE-Encrypt
            try (InputStream plainTextBlob = decryptBlob(guestIdentity, guestEmailData);) {
                MimeMessage plainTextMessage = new MimeMessage(Session.getDefaultInstance(new Properties()), plainTextBlob);
                MimeMessage reEncryptedMessage = pgpMimeService.encrypt(plainTextMessage, getRecipientKeysFrom(guestIdentity, plainTextMessage));
                addAdditionalHelpFiles(guestIdentity, reEncryptedMessage);
                reEncryptedMessage.saveChanges();
                reEncryptedMessage.writeTo(output);
            }
        } catch (Exception e) {
            throw GuardCoreExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.guest.impl.converters.GuestItemConverter#convert(com.openexchange.guard.guest.GuardGuestEmail)
     */
    @Override
    public GuestItemConverterResult convert(UserIdentity guestIdentity, InputStream guardGuestData, OutputStream output, String objectId) throws OXException {
        try {
            guardGuestData = Objects.requireNonNull(guardGuestData, "guardGuestData must not be null");
            InputStream bufferedInputStream = Streams.bufferedInputStreamFor(guardGuestData);
            if (isPGPBlob(bufferedInputStream)) {
                convertToPGPMime(guestIdentity, bufferedInputStream, output);
                return GuestItemConverterResult.CONVERTED;
            }
            return GuestItemConverterResult.NOT_CONVERTED_NOT_NECASSARY;
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        }
    }
}
