/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.servlets.keymanagement.actions;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseAction;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseWrapper;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.user.GuardCapabilities;
import com.openexchange.guard.user.OXUserService;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link DeleteKeyPairAction}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class DeleteKeyPairAction extends GuardServletAction {

    /* (non-Javadoc)
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {

        final Long keyId = ServletUtils.getLongParameter(request, Parameters.PARAM_KEYID,true);
        final GuardKeyService guardKeyService = Services.getService(GuardKeyService.class);
        final GuardKeys keyToDelete = guardKeyService.getKeys(userSession.getGuardUserId(), userSession.getGuardContextId(), keyId);

        if(keyToDelete != null) {
            if(keyToDelete.hasPrivateKey()) {
                //Brute Force protect the key deletion if the key ring has a password protected private key.
                final UserIdentity userIdentity = requireUserIdentityFrom(request, userSession.getUserId(), userSession.getContextId());
                GuardCapabilities  settings = Services.getService(OXUserService.class).getGuardCapabilieties(userSession.getContextId(), userSession.getUserId());
                if (settings.isNoDeletePrivate()) {
                    throw GuardCoreExceptionCodes.DISABLED_ERROR.create("Delete Private Key");
                }
                AntiAbuseWrapper bruteForceProtection = getAntiAbuseWrapper(keyToDelete, AntiAbuseUtils.getAllowParameters(request, userSession.getGuardUserId(),  userSession.getGuardContextId(), new String(userIdentity.getPassword())));
                bruteForceProtection.doAction(new AntiAbuseAction<Void>() {

                    @Override
                    public Void doAction() throws OXException {
                        guardKeyService.delete(keyToDelete, new String(userIdentity.getPassword()), userSession.getUserId(), userSession.getContextId());
                        return null;
                    }});
            }
            else {
                //Just delete if it is a public only key
                guardKeyService.delete(keyToDelete, null, userSession.getUserId(), userSession.getContextId());
            }
            ServletUtils.sendJsonOK(response, null);
        }
        else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND.create();
        }
    }
}
