/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.auth;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.json.JSONObject;
import com.google.gson.JsonObject;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;

/**
 * {@link DeleteRecoveryHandler}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class DeleteRecoveryHandler {

    /**
     * Delete password recovery for the user
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void deleteRecovery(HttpServletRequest request, HttpServletResponse response, int userId, int cid, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request, userId, cid);
        Api ap = new Api(cookie, request);
        // verify login
        if (!ap.verifyLogin()) {
            ServletUtils.sendNotAcceptable(response, "Must be logged in");
            return;
        }
        String password = JsonUtil.getStringFromJson(json, "password", true);

        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        GuardKeys key = keyService.getKeys(userId, cid);
        if(key == null) {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_IDS_ERROR.create(userId, cid);
        }

        // verify password
        if (!PGPUtil.verifyPassword(key.getPGPSecretKeyRing(), password, key.getSalt())) {
            ServletUtils.sendNotAcceptable(response, "Bad password");

            return;
        }

        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        ogKeyTableStorage.deleteRecovery(key);
        JSONObject resp = new JSONObject();
        resp.put("data", "OK");
        ServletUtils.sendHtmlOK(response, resp.toString());
    }

}
