/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.pgpcore.services;

import java.util.ArrayList;
import java.util.List;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;
import com.openexchange.tools.encoding.Base64;

/**
 * {@link PGPSignatureVerificationResultUtil} - A helper class around {@link PGPSignatureVerificationResult}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.2
 */
public class PGPSignatureVerificationResultUtil {

    private static final String VERIFIED_PROPERTY = "verified";
    private static final String DATE_PROPERTY = "signatureCreatedOn";
    private static final String MISSING_PROPERTY = "missing";
    private static final String ERROR_PROPERTY = "error";
    private static final String ISSUER_USERIDS_PROPERTY = "issuerUserIds";
    private static final String ISSUER_KEY_FINGERPRINT_PROPERTY = "issuerKeyFingerprint";
    private static final String ISSUER_KEY_ID_PROPERTY = "issuerKeyId";

    /**
     * Converts the given results into mime-header like String representations
     *
     * @param result The results
     * @return The string representations of the given results
     */
    public static String[] toHeaders (List<PGPSignatureVerificationResult> results) {
        List<String> stringResults = new ArrayList<String>();
        for(PGPSignatureVerificationResult result : results) {
            stringResults.add(toHeader(result));
        }
        String [] res = new String[stringResults.size()];
        return stringResults.toArray(res);
    }

    /**
     * Converts the given result into a mime-header like String representation
     *
     * @param result The result
     * @return The string representation of the given result
     */
    public static String toHeader (PGPSignatureVerificationResult result) {
        StringBuilder sb = new StringBuilder();
        sb.append(VERIFIED_PROPERTY + "=");
        sb.append(result.isVerified() ? "true" : "false");
        sb.append("; " + MISSING_PROPERTY + "=");
        sb.append(result.isMissing() ? "true" : "false");
        if (result.getSignature() != null) {
            sb.append("; " + DATE_PROPERTY + "=");
            sb.append(result.getSignature().getCreationTime().getTime());
        }
        if(result.getIssuerKey() != null) {
            sb.append("; " + ISSUER_KEY_ID_PROPERTY + "=");
            sb.append(result.getIssuerKey().getKeyID());

            sb.append("; " + ISSUER_KEY_FINGERPRINT_PROPERTY + "=");
            sb.append(PGPKeysUtil.getFingerPrint(result.getIssuerKey().getFingerprint()));

        }
        if (result.getIssuerUserIds() != null) {
            sb.append("; " + ISSUER_USERIDS_PROPERTY + "=");
            List<String> userIds = result.getIssuerUserIds();
            for(int i=0; i<userIds.size(); i++) {
                String userId = userIds.get(i);
                sb.append(userId);
                if(i != userIds.size()-1) {
                   sb.append(",");
                }
            }
        }
        if (result.getError() != null) {
            sb.append("; " + ERROR_PROPERTY + "=");
            sb.append(result.getError());
        }
        return Base64.encode(sb.toString());
    }
}
