/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.oxapi.streaming;

import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import org.apache.http.HttpEntity;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.HttpClientUtils;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.guard.oxapi.OxCookie;

/**
 * An OxDriveFileInputStream obtains input bytes from a remote file stored in OX Drive.
 *
 * By default OxDriveFileInputStream downloads the file in chunks of a fixed size.
 * This size can be changed by calling setChunkBufferSize().In Addition it is possible
 * to enable automatically buffer size adjustment by calling enableDynamicBuffering().
 * Beginning with a starting buffer size value, the size of the buffer will be increased
 * or decreased to the fit within a download time of 5-10 seconds per chunk.
 *
 * @author Benjamin Gruedelbach
 * @deprecated Do not use this class for streaming as it lacks performance;
 *             For reading chunks of bytes from a HTTP connection use the InputStream
 *             of the response's responseEntity directly (getEntity())
 */
@Deprecated
public class OxDriveFileInputStream extends InputStream {

    private byte buffer[];
    private long chunkSize = 10485760; 	//default buffer size of 10 MB
    private final long maxFullDownloadSize = 10485760; // default maximum size for full downloading files 10MB
    private boolean dynamicBuffering = false; //by default we do not adjust the chunk size of the downloads
    private long maxChunkSize = 0;
    private final int fileId;
    private final long fileSize;
    private long byteReadCounter = 0;
    private int bufferPointer = 0;
    private long requestCounter = 0;
    private final HttpClient httpClient;
    private HttpContext httpContext;
    private OxCookie cookie;
    private String userAgent;
    private final URI uri;
    private static Logger logger = LoggerFactory.getLogger(OxDriveFileInputStream.class);

    private enum ReadMode {
        //Reading a chunk of data in size of chunkSize
        ReadChunked,
        //Reading the complete file into the internal buffer
        ReadComplete
    }

    /**
     * Creates an OxDriveFileInputStream using a existing session
     * 
     * @param httpClient The HTTP client to use for downloading the file
     * @param cookie The session's cookie
     * @param userAgent The user Agent to use
     * @param scheme The protocol to use (usually "http" or "https")
     * @param host The host to use
     * @param port The port to use
     * @param folderId The ID of the file's folder
     * @param fileId The ID of the file to download
     * @param fileVersion The version of the file to download, or -1 to get the current version
     * @param fileSize The size of the file to download in bytes
     * @param session The session to use for downloading the file
     * @throws URISyntaxException If it was not possible to build the full
     *             URI from the given parameters
     */
    public OxDriveFileInputStream(HttpClient httpClient, OxCookie cookie, String userAgent, String scheme, String host, int port, String folderId, int fileId, int fileVersion, long fileSize, String session) throws URISyntaxException {
        this.httpClient = httpClient;
        this.cookie = cookie;
        this.userAgent = userAgent;
        this.fileSize = fileSize;
        this.fileId = fileId;
        this.uri = CreateUri(scheme, host, port, folderId, fileId, fileVersion, session);
    }

    /**
     * Creates an OxDriveFileInputStream using a existing HTTP context
     * 
     * @param httpClient The HTTP client to use for downloading the file
     * @param httpContext The HTTP context to use
     * @param scheme The protocol to use (usually "http" or "https")
     * @param host The host to use
     * @param port The port to use
     * @param folderId The ID of the file's folder
     * @param fileId The ID of the file to download
     * @param fileVersion The version of the file to download, or -1 to get the current version
     * @param fileSize The size of the file to download in bytes
     * @param session The session to use for downloading the file
     * @throws URISyntaxException If it was not possible to build the full
     *             URI from the give parameters
     */
    public OxDriveFileInputStream(HttpClient httpClient, HttpContext httpContext, String scheme, String host, int port, String folderId, int fileId, int fileVersion, long fileSize, String session) throws URISyntaxException {
        this.httpClient = httpClient;
        this.httpContext = httpContext;
        this.fileSize = fileSize;
        this.fileId = fileId;
        this.uri = CreateUri(scheme, host, port, folderId, fileId, fileVersion, session);
    }

    /**
     * Internal Method to build up the URI to use
     * 
     * @param scheme The protocol to use (usually "http" or "https")
     * @param host The host to use
     * @param port The port to use
     * @param folderId The ID of the file's folder
     * @param fileId The ID of the file to download
     * @param fileVersion The version of the file to download
     * @param session The session to use for downloading the file
     * @return The URI in the format [schema]://[host]:[port]/[request]
     * @throws URISyntaxException If it was not possible to build the full
     *             URI from the given parameters
     */
    //
    private URI CreateUri(String scheme, String host, int port, String folderId, int fileId, int fileVersion, String session) throws URISyntaxException {
        URIBuilder builder = new URIBuilder().setScheme(scheme).setHost(host).setPort(port).setPath("/ajax/files").setParameter("action", "document").setParameter("id", String.valueOf(fileId)).setParameter("folder", String.valueOf(folderId)).setParameter("session", String.valueOf(session));
        if (fileVersion > -1) {
            builder.setParameter("version", String.valueOf(fileVersion));
        }
        return builder.build();
    }

    /**
     * Internal method to read a chunk of data from the remote file and store
     * it in the local stream buffer
     */
    private void ReadNextIntoBuffer() throws IOException {
        ReadNextIntoBuffer(ReadMode.ReadChunked);
    }

    /**
     * Internal method to read data from the remote file and store
     * it in the local stream buffer
     * 
     * @param readMode The mode to read data
     */
    private void ReadNextIntoBuffer(ReadMode readMode) throws IOException {

        //Reading as long as we have more stuff to read
        if (byteReadCounter < fileSize) {
            HttpGet getRequest = new HttpGet(uri);
            getRequest.addHeader("accept", "application/json");
            getRequest.setHeader("Connection", "close");

            if (readMode == ReadMode.ReadChunked) {
                //Setting up the range header
                long upperRange = byteReadCounter + chunkSize - 1;
                if (upperRange > fileSize) {
                    upperRange = fileSize - 1;
                }

                String rangeHeader = String.format("bytes=%s-%s", String.valueOf(byteReadCounter), String.valueOf(upperRange));
                getRequest.setHeader("Range", rangeHeader);
            }

            //Sending request and retrieving a response
            long t0 = System.nanoTime();
            HttpResponse response = null;
            if (httpContext == null) {
                //using the given cookie
                getRequest.addHeader("Cookie", cookie.getCookieHeader());
                getRequest.setHeader("User-Agent", userAgent);
                logger.debug(getRequest.toString());
                response = httpClient.execute(getRequest);
            } else {
                //Using the given HTTP context
                logger.debug(getRequest.toString());
                response = httpClient.execute(getRequest, httpContext);
            }

            int returnStatus = response.getStatusLine().getStatusCode();
            if (returnStatus == 200 /* Status: OK */ || returnStatus == 206 /* Status: partial content */) {

                //Reading data
                HttpEntity responseEntity = response.getEntity();
                this.buffer = EntityUtils.toByteArray(responseEntity);
                long t1 = System.nanoTime();
                bufferPointer = 0;
                HttpClientUtils.closeQuietly(response);
                this.byteReadCounter += this.buffer.length;
                this.requestCounter++;

                double tDownloadTimeSec = ((t1 - t0) / 1000000d) / 1000d;
                double tDownloadTimeMSec = tDownloadTimeSec * 1000;
                long kbDownloaded = buffer.length / 1000;
                double kbPerSecond = kbDownloaded / tDownloadTimeSec;

                logger.debug(String.format("%d bytes read from remote in %f ms [%fkb/s]", buffer.length, tDownloadTimeMSec, kbPerSecond));

                if (this.dynamicBuffering) {
                    //dynamically adjust the chunk size of the next download of a chunk depending
                    //on the download duration of the current chunk
                    tDownloadTimeSec = Math.max(0.1, tDownloadTimeSec);
                    long newChunkSize = 0;
                    if (tDownloadTimeSec < 5 && this.buffer.length < fileSize && this.chunkSize < maxChunkSize) {
                        //increase the chunk size
                        int f = (int) (5 / tDownloadTimeSec);
                        newChunkSize = this.chunkSize * f;

                    } else if (tDownloadTimeSec > 10) {
                        //decrease the chunk size
                        double f = (tDownloadTimeSec / 10);
                        if (f > 0) {
                            newChunkSize = (int) (this.chunkSize / f);
                        }
                    }
                    //applying new chunk size if changed and does not exceed the maximum chunk size we can us
                    if (newChunkSize > 0) {
                        if (newChunkSize <= this.maxChunkSize) {
                            this.chunkSize = newChunkSize;
                            logger.debug(String.format("Setting new partial download transfer size to %s kb", String.valueOf(this.chunkSize / 1024)));
                        } else {
                            logger.debug(String.format("Could not set new partial download transfer size to %s kb " + "because it would exceed the maximum size of %s kb", String.valueOf(newChunkSize / 1024), String.valueOf(this.maxChunkSize / 1024)));
                        }
                    }
                }

            } else if (returnStatus == 416 /* Requested range not satisfiable */) {

                //The server does not support range requests, so we re-try the download in one part,
                //if it's size is not exceeding the maximum full download size
                logger.warn(String.format("HTTP Response 416: The server rejected chunk downloading file %s", this.fileId));
                if (this.fileSize <= this.maxFullDownloadSize) {
                    logger.warn(String.format("Retrying downloading conmplete file %s...", this.fileId));
                    ReadNextIntoBuffer(ReadMode.ReadComplete);
                } else {
                    String error = String.format("Cannot download file %s in one request because it's size %s exceeded the maximum of %s...", this.fileId, this.fileSize, this.maxFullDownloadSize);
                    throw new IOException(error);
                }
            } else {
                buffer = null;
                HttpClientUtils.closeQuietly(response);
                getRequest.releaseConnection();
                String errorMessage = "Error reading chunked data from remote drive. HTTP-Error: " + response.toString();
                throw new IOException(errorMessage);
            }
        } else {
            buffer = null;
        }

    }

    /**
     * Getter for the size of the next chunk to download
     * 
     * @return the size of the next chunk to download
     */
    public long getChunkBufferSize() {
        return this.chunkSize;
    }

    /**
     * Getter for the request counter
     * 
     * @return The current amount of request done
     */
    public long getRequestCounter() {
        return this.requestCounter;
    }

    /**
     * Sets the size of the internal buffer
     * 
     * @param bufferSize The size of the internal buffer
     */
    public void setChunkBufferSize(long bufferSize) {
        this.chunkSize = bufferSize;
    }

    /**
     * Returns the size of the remote file
     */
    public long getFileSize() {
        return this.fileSize;
    }

    /**
     * Enables dynamic buffering
     *
     * By default the buffer size of a downloaded chunk has a fix size.
     * By enabling dynamic buffering the chunk size of the downloaded chunk is adjusted dynamically
     * to get as much data within 3-5 seconds as possible.
     *
     * @param startBufferSize The size of the buffer in byte to start the first download with
     * @param maxChunkSize The maximum size of buffer in byte to use for a download
     */
    public void enableDynamicBuffering(long startBufferSize, long maxChunkSize) {
        this.chunkSize = startBufferSize;
        this.maxChunkSize = maxChunkSize;
        this.dynamicBuffering = true;
    }

    /**
     * Disables dynamic buffering
     *
     * After disabling dynamic buffering you might want to call
     * getChunkBufferSize() to check the current size of the buffer and/or
     * setChunkBufferSize() to define a new fixed buffer size
     */
    public void disableDynmaicBuffering() {
        this.dynamicBuffering = false;
    }

    @Override
    public int read() throws IOException {
        //Reading data from remote if the buffer is empty or everything has been read
        if (buffer == null || bufferPointer == buffer.length) {
            ReadNextIntoBuffer();
        }

        //return data at the current read position if we got some data
        if (buffer != null && buffer.length > 0) {
            byte ret = buffer[bufferPointer];
            bufferPointer++;
            return ret & 0xFF;
        }
        return -1;
    }

    @Override
    public int read(byte[] b) throws IOException {
        if (b.length > 0) {
            int bytesRead = 0;
            int bytesToRead = b.length;

            //we want to read n=b.length bytes
            while (bytesRead < b.length) {
                //Reading data from the remote if the buffer is empty or everything has been read
                if (this.buffer == null || this.bufferPointer == this.buffer.length) {
                    ReadNextIntoBuffer();
                }

                if (this.buffer != null && buffer.length > 0) {
                    int length = 0;
                    if (this.buffer.length - this.bufferPointer < bytesToRead) {
                        // We do not have enough bytes in the local buffer.
                        // We read as much from the local buffer as we have
                        length = this.buffer.length - this.bufferPointer;
                    } else {
                        // We do have enough bytes available in our local buffer.
                        length = bytesToRead;
                    }

                    //copy data into the output buffer
                    System.arraycopy(this.buffer, this.bufferPointer, b, bytesRead, length);
                    bytesToRead = bytesToRead - length;
                    bytesRead = bytesRead + length;
                    this.bufferPointer = this.bufferPointer + length;
                } else {
                    //Cancel, because we could not get more remote data
                    break;
                }
            }
            //return -1 per documentation/definition (if
            //nothing was read from the stream)
            //or the length of read bytes
            if (bytesRead == 0) {
                return -1;
            }
            return bytesRead;
            //return ((bytesRead == 0) ? -1 : bytesRead);
        }
        return 0;
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {

        byte[] buffer = new byte[len];
        int sizeRead = read(buffer);
        System.arraycopy(buffer, 0, b, off, sizeRead);
        return sizeRead;
    }
}
