/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.milter.service.impl;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.channels.WritableByteChannel;
import com.openexchange.guard.milter.internal.MilterConstants;
import com.openexchange.guard.milter.internal.Responder;
import com.openexchange.guard.milter.service.MilterResponse;

/**
 * {@link ActionResponse}
 *
 * Generates a response to send back to MTA to alter the original email
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class ActionResponse extends MilterResponse {

    private ByteBuffer dataToWrite;
    private int status;

    /* (non-Javadoc)
     * Send the response
     */
    @Override
    public void sendResponse(WritableByteChannel channel) throws IOException {
        Responder.sendPacket(channel, status, (ByteBuffer) dataToWrite.flip());
    }

    /**
     * Reset the response to create new
     */
    public void reset () {
        dataToWrite.clear();
        dataToWrite = null;
        status = 0;
    }

    /**
     * Add recipient to incoming email
     * @param rcp
     * @throws UnsupportedEncodingException
     */
    public void addRcpt(String rcp) throws UnsupportedEncodingException {
        addString(rcp);
        status = MilterConstants.RESPONSE_ADDRCPT;
    }

    /**
     * Remove recipient from incoming email
     * @param rcp
     * @throws UnsupportedEncodingException
     */
    public void delRcpt (String rcp) throws UnsupportedEncodingException {
        addString(rcp);
        status = MilterConstants.RESPONSE_DELRCPT;
    }

    /**
     * Add a header
     * @param header
     * @param value
     * @throws UnsupportedEncodingException
     */
    public void addHeader (String header, String value) throws UnsupportedEncodingException {
        dataToWrite = ByteBuffer.allocate(
            getStringLength(header) +
            getStringLength(value)
            );
        addString(header);
        addString(value);
        status = MilterConstants.RESPONSE_ADDHEADER;
    }

    /**
     * Change a header value.  Will change the first occurence
     * @param header
     * @param value
     * @throws UnsupportedEncodingException
     */
    public void changeHeader(String header, String value) throws UnsupportedEncodingException {
        dataToWrite = ByteBuffer.allocate(4 +
            getStringLength(header) +
            getStringLength(value)
            );
        dataToWrite.putInt(0);  // First location of header
        addString(header);
        addString(value);
        status = MilterConstants.RESPONSE_CHGHEADER;
    }

    /**
     * Replace the body with a new dataBuffer
     * @param dataBuffer
     */
    public void replBody(ByteBuffer dataBuffer) {
        dataToWrite = ByteBuffer.allocate(dataBuffer.limit());
        dataToWrite.put(dataBuffer);
        status = MilterConstants.RESPONSE_REPLBODY;
    }

    private void addString (String data) throws UnsupportedEncodingException {
        if (data != null) {
            byte[] datArray = data.getBytes("UTF-8");
            dataToWrite.put(datArray);
        }
        dataToWrite.put((byte) 0);
    }

    private int getStringLength (String string) {
        if (string == null) return 1;
        int len = string.length() + 1;
        return len;
    }
}
