/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.milter.server;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.SocketAddress;
import java.nio.channels.ServerSocketChannel;
import org.slf4j.Logger;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.milter.exceptions.MilterExceptionCodes;
import com.openexchange.guard.milter.osgi.MilterActivator;
import com.openexchange.guard.milter.osgi.Services;

/**
 * {@link MilterServer}
 *
 * Initiates the Milter Server.  Binds to address and port, then starts server thread
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class MilterServer {

    private ServerSocketChannel serverSocketChannel = null;
    Thread serverWatcher = null;

    Logger Log = org.slf4j.LoggerFactory.getLogger(MilterActivator.class);

    public void Startup () throws OXException {
        GuardConfigurationService config = Services.getService(GuardConfigurationService.class);
        if (config.getBooleanProperty(GuardProperty.enableEncryptIncoming) == false) {
            Log.info("Encrypt incoming not enabled, exiting milter server");
            return;
        }
        SocketAddress toBind = null;
        if (config.getProperty(GuardProperty.milterBindAddress).equals("*")) {
            toBind = new InetSocketAddress(config.getIntProperty(GuardProperty.milterBindPort));
        } else
            toBind = new InetSocketAddress(config.getProperty(GuardProperty.milterBindAddress), config.getIntProperty(GuardProperty.milterBindPort));
        try {
            this.serverSocketChannel = ServerSocketChannel.open();
            this.serverSocketChannel.configureBlocking(true);
            this.serverSocketChannel.socket().bind(toBind);
        } catch (IOException e) {
            Log.error("Problem starting milter server.  Check port configuration settings ", e);
            throw MilterExceptionCodes.UNABLE_STARTUP.create(e.getMessage());
        }
        serverWatcher = new Thread(
            new MilterServerWatcher(serverSocketChannel)
            );
        serverWatcher.setName("Milter Server");
        serverWatcher.start();

    }

    public void stop () {
        if (serverWatcher != null) serverWatcher.interrupt();
    }

}
