/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.servlets.keymanagement.actions;

import static com.openexchange.guard.servlets.keymanagement.actions.Parameters.PARAM_CREATE_IF_MISSING;
import static com.openexchange.guard.servlets.keymanagement.actions.Parameters.PARAM_EMAIL;
import static com.openexchange.guard.servlets.keymanagement.actions.Parameters.PARAM_KEYID;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.AccountCreationService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.RecipKeyService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.keymanagement.responses.RecipKeyInfoResponse;

/**
 * {@link GetRecipKeyAction} searches for other user's public keys and is also able to create new keys for users.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class GetRecipKeyAction extends GuardServletAction {

    /**
     * Gets a {@link RecipKey} by ID
     *
     * @param keyId The ID
     * @return The {@link RecipKey} for the given ID, or null if no such key was found
     * @throws OXException
     */
    private RecipKey getKey(long keyId) throws OXException {
        GuardKeys key = Services.getService(GuardKeyService.class).getKeys(keyId);
        if (key != null) {
            return new RecipKey(key);
        }
        return null;
    }

    /**
     * Gets a {@link RecipKey} by email
     *
     * @param userSession The user who searches for the key
     * @param email The email
     * @return The {@link RecipKey} for the given email, or null if no such key was found
     * @throws OXException
     */
    private RecipKey getKey(GuardUserSession userSession, String email) throws OXException {
        RecipKeyService recipKeyService = Services.getService(RecipKeyService.class);
        return recipKeyService.getRecipKey(userSession.getUserId(), userSession.getContextId(), email);
    }

    /**
     * Creates a new key
     *
     * @param userSession The user who requested a new key
     * @param recipKey The {@link RecipKey} to create
     * @return The update {@link RecipKey} object
     * @throws OXException
     */
    private RecipKey createKey(RecipKey recipKey) throws OXException {
        AccountCreationService accountCreationService = Services.getService(AccountCreationService.class);
        return accountCreationService.createUserFor(recipKey);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        final String email = ServletUtils.getStringParameter(request, PARAM_EMAIL);
        if(email != null) {
            GuardRatifierService ratifier = Services.getService(GuardRatifierService.class);
            ratifier.validate(email);
        }
        final Long keyId = ServletUtils.getLongParameter(request, PARAM_KEYID);
        final boolean createIfMissing = ServletUtils.getBooleanParameter(request, PARAM_CREATE_IF_MISSING, false);

        //If a keyID was specified it has to be a valid long value
        if(keyId == null && ServletUtils.hasParameter(request, PARAM_KEYID)) {
            throw GuardCoreExceptionCodes.INVALID_PARAMETER_VALUE.create(PARAM_KEYID);
        }

        //We need either key id or email
        if (keyId == null && Strings.isEmpty(email)) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create(PARAM_KEYID);
        }

        //It does not make sense to specify the create parameter while searching for a specific key ID
        if (keyId != null && createIfMissing) {
            throw GuardCoreExceptionCodes.PARAMETER_MISMATCH.create(PARAM_KEYID, PARAM_CREATE_IF_MISSING);
        }

        RecipKey recipKey = keyId != null ? getKey(keyId) : getKey(userSession, email);
        if (recipKey != null && recipKey.isNewKey() && createIfMissing) {
            //create missing key if desired by the client
            recipKey = createKey(recipKey);
        }

        if (recipKey != null) {
            ServletUtils.sendObject(response, new RecipKeyInfoResponse(recipKey, userSession.getUserId(), userSession.getContextId()));
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND.create();
        }
    }
}
