/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.servlets.guestreader;

import java.io.IOException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.osgi.Services;


/**
 * {@link GuestReaderServlet} - Redirecting GET requests to Guard's external guest reader  
 * <p>
 *   With OX Guard 2.4.0 the external guest reader is located in it's own package and not served by the servlet container anymore.
 *   We provide a redirect to the new location of the guest reader in order to access old guest-reader links.
 * </p> 
 * 
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GuestReaderServlet extends HttpServlet {
    
    private static final long serialVersionUID = 3165090161928784444L;
    private static final Logger logger = LoggerFactory.getLogger(GuestReaderServlet.class);
    private static final String FALLBACK_DEFAUL_READER_PATH = "/guard/reader/reader.html";
    private static final String LOCATION_HEADER = "Location";

    /**
     * Internal method read the location to redirect the incoming request
     * @param req the incoming request
     * @return the location to redirect the request
     * @throws OXException
     */
    private String getNewLocation(HttpServletRequest req) throws OXException {
        GuardConfigurationService configurationService = Services.getService(GuardConfigurationService.class);
        String readerPath = configurationService.getProperty(GuardProperty.externalReaderPath);
        //remove the domain part from the path
        if(readerPath.contains("/")) {
            readerPath = readerPath.substring(readerPath.indexOf("/"));
        }
        return readerPath;
    }
    
    /**
     * Internal method to read the query string from the incoming request
     * @param req the incoming request
     * @return the query string from the incoming request starting with "?", or and empty string, if the request does not contain a query string
     */
    private String getQueryString(HttpServletRequest req) {
        String queryString = req.getQueryString();
        return (queryString != null && !queryString.isEmpty()) ? "?" + queryString : "";
    }

    /**
     * Internal method to perform the redirect
     * @param req the incoming request
     * @param resp the response
     * @throws IOException
     */
    private void doRedirect(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        try {            
            String newLocation = getNewLocation(req) + getQueryString(req);
            logger.debug(String.format("Redirecting OX Guard reader request to %s", newLocation));
            resp.setStatus(HttpStatus.SC_MOVED_PERMANENTLY);
            resp.setHeader(LOCATION_HEADER, newLocation);
        }
        catch(Exception e) {
            logger.error("Error while redirecting to the guest reader location", e);
            String fallback = FALLBACK_DEFAUL_READER_PATH + getQueryString(req);
            logger.debug(String.format("Redirecting OX Guard reader request to the fallback location of %s", FALLBACK_DEFAUL_READER_PATH));
            resp.setStatus(HttpStatus.SC_MOVED_PERMANENTLY);
            resp.setHeader(LOCATION_HEADER, fallback);
        }
    }
    
    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {        
        doRedirect(req, resp);
    }
}
