/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.auth;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.MasterKeyService;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;

/**
 * {@link SecondaryEmailHandler}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class SecondaryEmailHandler {

    private static Logger LOG = LoggerFactory.getLogger(SecondaryEmailHandler.class);

    /**
     * Update secondary email address after verifying OxGuard password
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void changeSecondaryEmail(HttpServletRequest request, HttpServletResponse response, int userId, int cid, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request, userId, cid);
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {// verify logged into UI
            ServletUtils.sendNotAcceptable(response, "Not logged in");
            return;
        }
        String password = JsonUtil.getStringFromJson(json, "password", true);
        final String newemail = new EmailValidator().assertInput(JsonUtil.getStringFromJson(json, "email", true), "email");
        GuardKeyService keyService = Services.getService(GuardKeyService.class);

        if (antiabuse.blockLogin(AntiAbuseUtils.getAllowParameters(request, userId, cid, password))) {
            ServletUtils.sendNotAcceptable(response, "Lockout");
            return;
        }
        GuardKeys keys = keyService.getKeys(userId, cid);
        if(keys != null) {
            if (PGPUtil.verifyPassword(keys.getPGPSecretKeyRing(), password, keys.getSalt())) {
                if (updateEmail(userId, cid, newemail)) {
                    JSONObject resp = new JSONObject();
                    resp.put("data", "OK");
                    ServletUtils.sendHtmlOK(response, resp.toString());
                    antiabuse.report(AntiAbuseUtils.getReportParameter(true, request, userId,  cid, password));
                } else {
                    LOG.info("Problem updating secondary email");
                    ServletUtils.sendNotAcceptable(response, "Fail");
                }
            } else {
                antiabuse.report(AntiAbuseUtils.getReportParameter(false, request, userId,  cid, password));
                LOG.info("Bad password for changing secondary email");
                ServletUtils.sendNotAcceptable(response, "Bad password");
                return;
            }
        }
        else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_IDS_ERROR.create(userId, cid);
        }
    }

    public String getSecondaryEmail(int userid, int cid) throws Exception {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        GuardKeys userKeys = ogKeyTableStorage.getCurrentKeyForUser(userid, cid);
        String encr_answer = userKeys.getAnswer();
        if (encr_answer == null) {
            return null;
        }
        return Services.getService(MasterKeyService.class).getRcDecrypted(encr_answer, "e", userKeys.getMasterKeyIndex());

    }

    private boolean updateEmail(int userid, int cid, String email) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        return (keyService.storeQuestion(userid, cid, "e", email));
    }


}
