/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.keymanagement.services.impl;

import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.MasterKey;
import com.openexchange.guard.keymanagement.services.KeyRecoveryService;
import com.openexchange.guard.keymanagement.services.MasterKeyService;
import com.openexchange.guard.keymanagement.services.impl.exceptions.GuardKeyTableExceptionCodes;
import com.openexchange.guard.keymanagement.services.impl.osgi.Services;
import com.openexchange.guard.user.GuardCapabilities;
import com.openexchange.guard.user.OXUserService;
import com.openexchange.server.ServiceLookup;

/**
 * {@link KeyRecoveryServiceImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class KeyRecoveryServiceImpl implements KeyRecoveryService {

    ServiceLookup services;

    /**
     * Initializes a new {@link KeyRecoveryServiceImpl}.
     *
     * @param configurationService The configuration service
     * @param userService The user service
     */
    public KeyRecoveryServiceImpl(ServiceLookup services) {
        this.services = services;
    }

    private GuardConfigurationService getConfigService() throws OXException {
        return services.getServiceSafe(GuardConfigurationService.class);
    }


    private String createRecoveryInternal(GuardKeys keys, String password) throws OXException {
        MasterKeyService masterKeyService = services.getServiceSafe(MasterKeyService.class);
        MasterKey mkey = masterKeyService.getMasterKey(keys.getMasterKeyIndex(), true);
        String recovery = "";
        GuardCipherService aesCipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        String encrPass = aesCipherService.encrypt(password, mkey.getRC() + (keys.getPublicKey() == null ? "" : Integer.toString(keys.getEncodedPublic().hashCode())), keys.getSalt());
        GuardCipherService rsaCipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.RSA);
        recovery = rsaCipherService.encrypt(encrPass, mkey.getMasterPublic());


        if (Strings.isEmpty(recovery) && keys.getUserid() > 0) {  // make sure not for system email
            throw GuardKeyTableExceptionCodes.UNABLE_TO_CREATE_RECOVERY.create(keys.getUserid(), keys.getContextid(), keys.getEmail());
        }
        return recovery;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.KeyRecoveryService#checkCreateRecovery(int, int)
     */
    @Override
    public boolean checkCreateRecovery(int contextId, int userId) throws OXException {
        if (getConfigService().getBooleanProperty(GuardProperty.noRecovery, userId, contextId)) {
            // If global no Recovery then return false.
            return false;
        }
        if(contextId < 0) {
            return true;
        }

        // get individual settings
        OXUserService userService = services.getServiceSafe(OXUserService.class);
        GuardCapabilities userCapabilities = userService.getGuardCapabilieties(contextId, userId);
        return !userCapabilities.isNorecovery();
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.KeyRecoveryService#createRecovery(com.openexchange.guard.keymanagement.commons.GuardKeys, java.lang.String)
     */
    @Override
    public String createRecovery(GuardKeys keys, String password) throws OXException {
        if (Strings.isEmpty(password)) {
            return "-1";
        }
        password = CipherUtil.getSHA(password, keys.getSalt());
        return createRecoveryInternal(keys, password);
    }
}
