/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * © 2016 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

define('oxguard/pgp/uploadkeys', [
    'io.ox/core/extensions',
    'io.ox/core/yell',
    'io.ox/backbone/views/modal',
    'io.ox/core/notifications',
    'oxguard/core/passwordView',
    'oxguard/oxguard_core',
    'gettext!oxguard'
], function (ext, yell, ModalView, notify, PasswordView, core, gt) {

    'use strict';

    var uploading = false;

    // Uploads to personal key chain
    // upload(files) for uploading public key only
    // upload (file, private, pgp Password, Guard password) for uploading private key
    function upload(files, priv, pgp, guard) {
        var deferred = $.Deferred();
        if (uploading) return $.Deferred().reject();
        var formData = new FormData();
        for (var l = 0; l < files.length; l++) {
            if (!validFileType(files[l])) {
                deferred.reject();
                return deferred;
            }
            if (files[l] !== undefined) { formData.append('key' + l, files[l]); }
        }
        if (priv) {
            formData.append('keyPassword', pgp);
            formData.append('newPassword', guard);
        }
        uploading = true;
        var url = ox.apiRoot + '/oxguard/keys?action=upload&respondWithJSON=true&session=' + ox.session;
        $.ajax({
            url: url,
            type: 'POST',
            data: formData,
            processData: false, // tell jQuery not to process the data
            contentType: false, // tell jQuery not to set contentType
            success: function (data) {
                if (data === null) data = '';
                if (core.checkJsonOK(data)) {
                    try {
                        if (data.data && data.data.keyRings) {
                            var keys = data.data.keyRings;
                            var added = gt('Added keys: \r\n');
                            for (var i = 0; i < keys.length; i++) {
                                var key = keys[i];
                                if (key.privateRing && key.privateRing.keys) {
                                    var privateKeys = key.privateRing.keys;
                                    added = added + gt('Private keys: ');
                                    for (var p = 0; p < privateKeys.length; p++) {
                                        var pKey = privateKeys[p];
                                        added = added + pKey.fingerPrint + '\r\n';
                                        for (var d = 0; d < pKey.userIds.length; d++) {
                                            added = added + pKey.userIds[d] + ' ';
                                        }
                                        added = added + '\r\n';
                                    }
                                } else {
                                    var publicKeys = key.publicRing.keys;
                                    added = added + gt('Public keys: ');
                                    for (var k = 0; k < publicKeys.length; k++) {
                                        var pubKey = publicKeys[k];
                                        added = added + pubKey.fingerPrint + '\r\n';
                                        for (var j = 0; j < pubKey.userIds.length; j++) {
                                            added = added + pubKey.userIds[j] + ' ';
                                        }
                                        added = added + '\r\n';
                                    }
                                }
                            }
                            notify.yell('success', added);
                        }
                    } catch (e) {
                        console.log(e);
                    }
                    $('input[name="publicKey"]').val('');
                    deferred.resolve('OK');
                } else {
                    deferred.reject(data.error);
                }
                uploading = false;

            },
            error: function (XMLHttpRequest) {
                $('input[name="publicKey"]').val('');
                uploading = false;
                var resp = XMLHttpRequest.responseText.trim();
                switch (resp.trim()) {
                    case 'Bad UID':
                        notify.yell('error', gt('Failed to upload') + '\r\n' + gt('Invalid Email Address'));
                        break;
                    case 'Bad ID':
                        notify.yell('error', gt('The public key must have your primary email as a user ID'));
                        break;
                    default:
                        notify.yell('error', gt('Failed to upload') + ' ' + XMLHttpRequest.responseText.trim());
                        break;
                }
                deferred.reject(XMLHttpRequest.responseText.trim());
            }
        });
        return (deferred);
    }

    // Verify ascii data contains key before upload
    function validateKeyData(key) {
        if (!/(-----BEGIN PGP PRIVATE KEY BLOCK-----|-----BEGIN PGP PUBLIC KEY BLOCK-----)[\d\w\r\n:. +/=]+(-----END PGP PRIVATE KEY BLOCK-----|-----END PGP PUBLIC KEY BLOCK-----)/i.test(key)) {
            notify.yell('error', gt('Does not appear to be a valid key file type'));
            return false;
        }
        return true;
    }

    // Upload ascii key data
    function uploadPublicKey(key) {
        var deferred = $.Deferred();
        if (!validateKeyData(key)) {
            deferred.reject();
            return deferred;
        }
        var formData = new FormData();
        formData.append('key', key);
        var url = ox.apiRoot + '/oxguard/keys?action=upload&session=' + ox.session;
        $.ajax({
            url: url,
            type: 'POST',
            data: formData,
            processData: false, // tell jQuery not to process the data
            contentType: false, // tell jQuery not to set contentType
            success: function (data) {
                if (core.checkJsonOK(data)) {
                    notify.yell('success', gt('Keys uploaded'));
                    $('#refreshuserkeys').click();
                    deferred.resolve();
                }
            },
            error: function (XMLHttpRequest) {
                $('input[name="publicKey"]').val('');
                uploading = false;
                var resp = XMLHttpRequest.responseText.trim();
                switch (resp.trim()) {
                    case 'Bad UID':
                        notify.yell('error', gt('Failed to upload') + '\r\n' + gt('Invalid Email Address'));
                        break;
                    case 'Bad ID':
                        notify.yell('error', gt('The public key must have your primary email as a user ID'));
                        break;
                    default:
                        notify.yell('error', gt('Failed to upload') + ' ' + XMLHttpRequest.responseText.trim());
                        break;
                }
                deferred.reject(XMLHttpRequest.responseText.trim());
            }
        });
        return deferred;
    }

    function uploadExternalKey(files) {
        var deferred = $.Deferred();
        if (uploading) return $.Deferred().reject();
        var formData = new FormData();
        for (var l = 0; l < files.length; l++) {
            if (files[l] !== undefined) {
                if (!validFileType(files[l])) {
                    deferred.reject();
                    return deferred;
                }
                formData.append('key' + l, files[l]);
            }
        }
        uploading = true;
        var url = ox.apiRoot + '/oxguard/keys?action=uploadExternalPublicKey&respondWithJSON=true&session=' + ox.session;
        $.ajax({
            url: url,
            type: 'POST',
            data: formData,
            processData: false, // tell jQuery not to process the data
            contentType: false, // tell jQuery not to set contentType
            success: function (data) {
                if (data === null) data = '';
                if (core.checkJsonOK(data)) {
                    deferred.resolve('ok');
                } else {
                    deferred.reject();
                }
                uploading = false;

            },
            error: function (XMLHttpRequest) {
                $('input[name="publicKey"]').val('');
                deferred.reject(XMLHttpRequest.responseText.trim());
                uploading = false;
            }
        });
        return (deferred);
    }

    function validFileType(file) {
        if (!/^.*\.(txt|pgp|gpg|asc)$/i.test(file.name)) {
            console.error('bad file type ' + file.name);
            notify.yell('error', gt('Does not appear to be a valid key file type'));
            return false;
        }
        return true;
    }

    function createPasswordPrompt(files, def) {
        return new ModalView({
            async: true,
            point: 'oxguard/pgp/uploadKeysPasswordPrompt',
            title: gt('Upload Private Keys'),
            width: 640,
            enter: 'ok',
            focus: '#pgppassword'
        })
        .inject({
            doUpload: function () {
                return doUploadPrivate(files);
            }
        })
        .build(function () {
        })
        .addButton({ label: gt('OK'), action: 'ok' })
        .addCancelButton()
        .on('ok', function () {
            var dialog = this;
            this.doUpload().done(function () {
                def.resolve();
                dialog.close();
            })
            .fail(function (e) {
                yell(e);
                $('#uploaderror').text(e);
                dialog.idle();
            });
        })
        .open();
    }

    ext.point('oxguard/pgp/uploadKeysPasswordPrompt').extend(
        {
            index: 100,
            id: 'passwordPrompts',
            render: function () {

                this.$body.append(
                    createPrompts()
                );
            }
        });

    function createPrompts() {
        var explain = $('<div><p>' + gt('Please enter passwords for the upload') + '</p>');
        var passdiv = $('<div>').addClass('row-fluid');
        var newogpassword = new PasswordView.view({ 'id': 'guardpassword', 'class': 'password_prompt', 'validate': true }).getProtected();
        var newogpassword2 = new PasswordView.view({ 'id': 'guardpassword2', 'class': 'password_prompt', 'validate': true }).getProtected();
        var pgpPassword = new PasswordView.view({ 'id': 'pgppassword', 'class': 'password_prompt' }).getProtected();
        var hint = $('<td>');
        var pw1 = $('<tr>').append($('<td class="pw">').append('<label for="guardpassword">' + gt('Enter new password for the key:') + '</label>')).append($('<td>').append(newogpassword));
        var pw2 = $('<tr>').append($('<td class="pw">').append('<label for="guardpassword2">' + gt('Confirm Password:') + '</label>')).append($('<td>').append(newogpassword2));
        var noSaveWorkAround = $('<input style="display:none" type="text" name="dontremember"/><input style="display:none" type="password" name="dontrememberpass"/>');
        var table = $('<table class="og_password_prompt"/>')
        .append(noSaveWorkAround)
        .append($('<tr>').append($('<td class="pw"><label for="pgppassword">' + gt('Private key password for the key you are uploading:') + '</label></td>')).append($('<td>').append(pgpPassword)))
        .append(pw1)
        .append(pw2)
        .append($('<tr>').append('<td>').append(hint));
        var errormessage = $('<span id="uploaderror" style="color:red;"></span>');
        passdiv.append(table);
        require(['oxguard/core/passwords'], function (pass) {
            pass.passwordCheck(newogpassword, hint);
            pass.passwordCompare(newogpassword2, newogpassword, hint);
        });
        return explain.append(passdiv).append(errormessage);
    }

    function doUploadPrivate(files) {
        var deferred = $.Deferred();
        var pgp = $('#pgppassword').val();
        var guard = $('#guardpassword').val();
        var guard2 = $('#guardpassword2').val();
        if (guard !== guard2) {
            deferred.reject(gt('Passwords not equal'));
            return deferred;
        }
        if (window.oxguarddata.settings.min_password_length !== undefined) {  // Check min password length
            if (guard.trim().length < window.oxguarddata.settings.min_password_length) {
                deferred.reject(gt('New password must be at least %s characters long', window.oxguarddata.settings.min_password_length));
                return deferred;
            }
        } else if (guard.trim().length < 2) { // If no min defined, make sure at least 2 characters there
            uploadPrivate(files);
            return deferred;
        }
        upload(files, true, pgp, guard)
        .done(function () {
            deferred.resolve('OK');
        })
        .fail(function (e) {
            console.log(e);
            deferred.reject(e ? e.trim() : '');
            handleFail(e, notify);
        });
        return deferred;
    }

    function uploadPrivate(files) {
        var deferred = $.Deferred();
        for (var i = 0; i < files.length; i++) {
            if (!validFileType(files[i])) {
                deferred.reject();
                return deferred;
            }
        }
        createPasswordPrompt(files, deferred);
        return deferred;
    }

    function handleFail(e, notify) {
        if (e === 'duplicate') {
            notify.yell('error', gt('This key already exists'));
            return;
        }
        if (e === 'Bad pgp') {
            notify.yell('error', gt('Bad PGP Password'));
            return;
        }
        if (e === 'Bad password') {
            notify.yell('error', gt('Bad Password'));
            return;
        }
        if (e === 'Bad ID') {
            notify.yell('error', gt('The private key must have your primary email as a user ID'));
            return;
        }
        if (e === 'Bad Merge Password') {
            notify.yell('error', gt('Must use same password as existing key to merge'));
            return;
        }
        notify.yell('error', gt('Problem uploading Private Key ') + '\r\n' + e);
    }

    return {
        upload: upload,
        uploadPrivate: uploadPrivate,
        uploadPublicKey: uploadPublicKey,
        uploadExternalKey: uploadExternalKey
    };

});
