/* All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */

define('oxguard/mail/checker', [
    'io.ox/core/extensions',
    'io.ox/core/extPatterns/links',
    'oxguard/core/og_http',
    'oxguard/oxguard_core',
    'io.ox/core/capabilities'
], function (ext, links, http, core, capabilities) {

    'use strict';

    var firstload = true;

    // Do the check if the email is PGP email.  Check for inline, MIME, or signature
    function checkPGP(baton, location) {
        if (baton.data.security && baton.data.security.decrypted) {
            if (window.oxguarddata.settings.oxguard === false && !capabilities.has('guest')) { // If not guard user, display upsell if any
                var loc = baton.view.$el;
                if (loc.find('.guardUpsell').length < 1) { // Make sure not already drawn
                    require(['oxguard/mail/upsell'], function (upseller) {
                        upseller.upsell(loc);
                    });
                }
            }
            return; // If already decoded
        }
        if (baton.view.model === null) {
            return;
        }
        var headers = baton.view.model.get('headers');
        if (headers === undefined) { // not yet loaded
            return;
        }
        // Init
        if (baton.data.headers === undefined) { // If email wasn't fully loaded into baton, do so now.
            baton.data = baton.view.model.toJSON();
        }
        var mail = baton.data;
        if (mail.results !== undefined) return;

        if (headers['X-OXGUARD-GUEST'] !== undefined) {
            baton.view.model.set('pgp_found', true);
            pgpAttach(baton);
            pgpFound.call(this, baton);
        }
        // Main check if Guard email
        if (baton.view.model.get('security') && baton.view.model.get('security').decrypted) return;  // Bug 53622, loop in mobile view
        if (mail.security_info && mail.security_info.encrypted) {
            baton.view.model.set('pgp_found', true);
            pgpAttach(baton);
            pgpFound.call(this, baton, location);
        } else if (mail.security_info && mail.security_info.signed) {
            if (!mail.security && !baton.signatureCheck) pullAgain(false, baton);  // If we don't already have result, reload
            baton.signatureCheck = true;  // avoid loops
        }
    }

    function showError(data, baton) {
        require(['oxguard/oxguard_core'], function (core) {
            core.removeAuth();  // wipe auth
            core.showError(data);
            if (!baton.signatureCheck) baton.view.redraw();  // avoid loop
        });
    }

    function pullAgain(decrypt, baton) {
        require(['io.ox/mail/api'], function (api) {
            var options = {
                cache: false
            };
            var obj = {
                id: baton.data.id,
                folder_id: baton.data.folder_id
            };
            if (decrypt) {
                obj.decrypt = 'true';
            } else {
                obj.verify = 'true';
            }
            api.get(obj, options)
            .then(function (data) {
                data = removeSignatureFile(data); // remove verified signatures
                if (baton.sessionKey) { // If local decrypt, save the sessionKey for attachments, etc
                    data.security.sessionKey = baton.sessionKey;
                }
                var origData = _.extend(baton.data, {});
                baton.model.set(data);
                origData.view = baton.view;
                baton.model.set('origMail', origData);
                // Update the pool mail with the attachment and security data of the email.  Otherwise will
                // be overwritten with changes in listViews, sort order, etc
                var stored = api.pool.get('detail').get(_.cid(data));
                if (stored) {
                    stored.set('guardIcons', _.pick(data, 'attachment', 'security'));
                }
                delete (baton.view.attachmentView); // Need to redo the attachment list
                baton.view.redraw();
            }, function (data) {
                showError(data, baton);
            });
        });
    }

    // Remove signature.asc files from attachments if signatures verified
    function removeSignatureFile(data) {
        if (data.security && data.security.signatures) {
            var verified = true;
            data.security.signatures.forEach(function (d) {
                verified = verified && d.verified;
            });
            if (verified) {
                var newAttachments = [];
                data.attachments.forEach(function (att) {
                    if (att.content_type !== 'application/pgp-signature') {
                        newAttachments.push(att);
                    }
                });
                data.attachments = newAttachments;
            }
        }
        return data;
    }

    function busy(location) {
        $(location).busy();
    }

    // PGP found, do send for decode
    function pgpFound(baton, location, ignoreLocal) {
        if (!ignoreLocal && core.hasLocalKeys()) {
            require(['pgp_local/localHandler'], function (localMail) {
                localMail.mailFound(baton, location);
            });
            return;
        }
        var goFunction = function () {
            pullAgain(true, baton);
        };
        checkLoaded().then(function () {
            checkSession().then(function (auth) {
                require(['oxguard/mail/oxguard_mail_password'], function (oxmail) {
                    if (auth && auth.length > 10) {  // If auth code in session, load decrypted
                        pullAgain(true, baton);
                    } else if (window.oxguarddata.passcode === null) {
                        $(location).replaceWith(oxmail.passwordPrompt(baton, false, location, goFunction));
                        baton.view.$el.find('.attachments').hide();
                        if (!_.device('ios')) {
                            baton.view.$el.find('#oxgrpass').focus();
                        }
                        baton.view.$el.show();
                    } else {
                        busy(location, oxmail.getId(baton));
                        checkauth(goFunction, baton, undefined);

                    }
                });
            });
        });
    }

    function checkSession() {
        var def = $.Deferred();
        if (!firstload) {
            def.resolve();
            return def;
        }
        require(['oxguard/oxguard_core'], function (core) {
            core.checkAuth()
            .done(function (data) {
                firstload = false;
                def.resolve(data.auth);
            })
            .fail(function () {
                firstload = false;
                def.resolve();
            });
        });
        return def;
    }

    // Timeout loop waiting for the Guard backend data to be loaded
    function waitForLoad(def) {
        if (window.oxguarddata.loaded) {
            def.resolve();
            return;
        }
        setTimeout(function () {
            waitForLoad(def);
        }, 500);
    }

    // Check if Guard backend data loaded.  If not, start wait loop
    function checkLoaded() {
        var def = $.Deferred();
        if (window.oxguarddata.loaded) {
            def.resolve();
        } else {
            waitForLoad(def);
        }
        return def;
    }

    function checkauth(go, baton, index) {
        if (window.oxguarddata.passcode.indexOf('Password Needed') > -1) { // If no password for user has been defined, open dialog
            require(['oxguard/core/tempPassword'], function (core) {
                core.createOxGuardPasswordPrompt(baton, go, '', undefined, index, location);
            });

            return;
        }
        if (window.oxguarddata.passcode.indexOf('No Key') > -1) { // If received a first encrypted email while already logged in, will still have 'no key' in passcode
            require(['oxguard/core/tempPassword', 'oxguard/oxguard_core'], function (password, core) {
                core.auth(ox.user_id, '') // Try to reload authorization to see if keys now exist
                .done(function (data) {
                    var oxguarddata = _.extend(window.oxguarddata, data);
                    if (data.auth === 'No Key') {
                        baton.model.unset('security_info');
                        baton.view.redraw();
                        return;  // Still no key, non guard generated email?
                    }
                    window.oxguarddata = oxguarddata; // Update data
                    password.createOxGuardPasswordPrompt(baton, go, '', undefined, index, location);  // Create new password
                });

            });
            return;
        }
        go(baton, '', index, window.oxguarddata.passcode);
    }

    function pgpAttach(baton) {
        baton.data.PGP = true;
    }

    return {
        checkPGP: checkPGP,
        pullAgain: pullAgain,
        showError: showError,
        pgpFound: pgpFound
    };
});
