/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.storage.s3.internal;

import java.io.ByteArrayInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.client.builder.AwsClientBuilder.EndpointConfiguration;
import com.amazonaws.regions.Regions;
import com.amazonaws.services.s3.AmazonS3Client;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.amazonaws.services.s3.S3ClientOptions;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.Region;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.dataobject.EncryptedObject;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.storage.AbstractLegacyStorage;
import com.openexchange.guard.storage.Storage;

/**
 * Class for saving encrypted objects to and from an S3 object store.
 *
 * @author benjamin.otterbach@open-xchange.com
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class S3LegacyStorageImpl extends AbstractLegacyStorage {

    private static Logger logger = LoggerFactory.getLogger(S3LegacyStorageImpl.class);

    private AmazonS3Client s3Client;

    private final GuardConfigurationService guardConfigService;

    /**
     * Initialises a new {@link S3LegacyStorageImpl}.
     *
     * @throws OXException
     */
    public S3LegacyStorageImpl(GuardConfigurationService configService) throws OXException {
        super();
        this.guardConfigService = configService;
        initialiseClient(configService);
    }

    /**
     * Initialises the S3 Client
     *
     * @throws OXException
     */
    private void initialiseClient(GuardConfigurationService guardConfigService) throws OXException {
        String accessKey = guardConfigService.getProperty(GuardProperty.accessKey);
        String secret = guardConfigService.getProperty(GuardProperty.secretKey);
        String region = guardConfigService.getProperty(GuardProperty.region);
        String bucketName = guardConfigService.getProperty(GuardProperty.bucketName);
        String endpoint = guardConfigService.getProperty(GuardProperty.endPoint);

        BasicAWSCredentials credentials = new BasicAWSCredentials(accessKey, secret);

        /*
         * Instantiate client
         */
        ClientConfiguration clientConfiguration = new ClientConfiguration();
        String signerOverride = guardConfigService.getProperty(GuardProperty.signerOverride);
        if (signerOverride != null && !signerOverride.isEmpty()) {
            clientConfiguration.setSignerOverride(signerOverride);
        }
        AmazonS3ClientBuilder builder = AmazonS3ClientBuilder.standard()
            .withCredentials(new AWSStaticCredentialsProvider(credentials))
            .withClientConfiguration(clientConfiguration);

        /*
         * Configure client
         */
        builder.setPathStyleAccessEnabled(true);

        if (Strings.isNotEmpty(endpoint)) {
            builder.setEndpointConfiguration(new EndpointConfiguration(endpoint, region));
        } else {
            try {
                builder.withRegion(Regions.fromName(region));
            } catch (IllegalArgumentException e) {
                logger.error("Error at s3Client setRegion", e);
            }
        }

        s3Client = (AmazonS3Client) builder.build();

        /*
         * Check if configured S3 bucket exists, otherwise create.
         */
        if (!s3Client.doesBucketExist(bucketName)) {
            s3Client.createBucket(bucketName);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#writeObj(java.lang.String, java.lang.String, byte[])
     */
    @Override
    public String writeObj(String directoryPrefix, String ObjId, byte[] data) throws OXException {
        String filename = null;

        /*
         * prepare upload
         */
        try {
            //            if (data.length < UPLOAD_BUFFER_SIZE) {
            /*
             * whole file fits into buffer (this includes a zero byte file), upload directly
             */
            ByteArrayInputStream dataStream = null;
            try {
                String key = directoryPrefix + Storage.DELIMITER + ObjId;
                dataStream = new ByteArrayInputStream(data);
                ObjectMetadata metadata = new ObjectMetadata();
                metadata.setContentLength(data.length);
                s3Client.putObject(guardConfigService.getProperty(GuardProperty.bucketName), key, dataStream, metadata);
                filename = key;
            } finally {
                if (dataStream != null) {
                    dataStream.close();
                }
            }
        } catch (Exception e) {
            logger.error("Error while uploading file to S3 store", e);
            throw GuardCoreExceptionCodes.IO_ERROR.create(e);
        }
        return filename;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#getEncrObj(java.lang.String, java.lang.String)
     */
    @Override
    public EncryptedObject getEncrObj(String directoryPrefix, String ObjId) throws OXException {
        String bucketName = "";
        try {
            bucketName = guardConfigService.getProperty(GuardProperty.bucketName);
            InputStream in = s3Client.getObject(bucketName, directoryPrefix + Storage.DELIMITER + ObjId).getObjectContent();
            EncryptedObject obj = new EncryptedObject(in);
            in.close();
            return (obj);
        } catch (FileNotFoundException e) {
            logger.error("File \"" + directoryPrefix + Storage.DELIMITER + ObjId + "\" not found in bucket \"" + bucketName + "\"");
            EncryptedObject obj = new EncryptedObject();
            obj.setItemID("not found");
            return (obj);
        } catch (Exception e) {
            logger.error("Error getEncrObj", e);
            return (null);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#getObjectStream(java.lang.String, java.lang.String)
     */
    @Override
    public InputStream getObjectStream(String directoryPrefix, String ObjId) throws OXException {
        try {
            InputStream in = s3Client.getObject(guardConfigService.getProperty(GuardProperty.bucketName), directoryPrefix + Storage.DELIMITER + ObjId).getObjectContent();
            return (in);
        } catch (Exception e) {
            logger.error("Error getEncrObj", e);
            return (null);
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#getObjectStream(java.lang.String)
     */
    @Override
    public InputStream getObjectStream(String location) throws OXException {
        try {
            InputStream in = s3Client.getObject(guardConfigService.getProperty(GuardProperty.bucketName), location).getObjectContent();
            return (in);
        } catch (Exception e) {
            logger.error("Error getEncrObj", e);
            return (null);
        }
    }


    /* (non-Javadoc)
     * @see com.openexchange.guard.storage.Storage#objectExists(java.lang.String)
     */
    @Override
    public boolean objectExists(String location) throws OXException {
        /* not implemented yet*/
        return true;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#deleteObj(java.lang.String)
     */
    @Override
    public void deleteObj(String location) throws OXException {
        s3Client.deleteObject(guardConfigService.getProperty(GuardProperty.bucketName), location);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.storage.Storage#deleteEncrObj(java.lang.String)
     */
    @Override
    public void deleteEncrObj(String location) throws OXException {
        deleteObj(location);
    }
}
