/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.servlets.keymanagement.actions;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.KeyImportService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.fileupload.FileUploadHandler;
import com.openexchange.guard.servlets.keymanagement.responses.GuardKeyInfoCollectionResponse;
import com.openexchange.pgp.keys.parsing.KeyRingParserResult;
import com.openexchange.pgp.keys.parsing.PGPKeyRingParser;

/**
 * {@link UploadKeysAction} allows to upload custom, client created, public and private keys.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class UploadKeysAction extends GuardServletAction {

    private static final int MAX_KEY_FILE_UPLOAD_SIZE = 1000000; /* 1 MB */
    private static final String KEY_PARAMETER_NAME= "key";
    private static final String KEY_PASSWORD_PARAMETER_NAME = "keyPassword";
    private static final String NEW_PASSWORD_PARAMETER_NAME = "newPassword";

    /**
     * Internal method to remove duplicated key rings
     *
     * @param importedKeyRings The list of keys rings
     * @return The list where duplicates have been removed
     */
    private Collection<GuardKeys> removeDuplicatedImportedKeyResults(Collection<GuardKeys> importedKeyRings) {

        Collection<GuardKeys> ret = new ArrayList<GuardKeys>(importedKeyRings.size());
        Collection<Long> recognizedIds = new ArrayList<Long>(importedKeyRings.size());

        importedKeyRings.forEach(key -> {
            if(!recognizedIds.contains(new Long(key.getPGPPublicKey().getKeyID()))){
                recognizedIds.add(new Long(key.getPGPPublicKey().getKeyID()));
                ret.add(key);
            }
        });

        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {

        //Parsing
        FileUploadHandler fileUploadHandler = new FileUploadHandler(Services.getService(GuardConfigurationService.class));
        List<PGPSecretKeyRing> parsedSecretKeyRings = null;
        List<PGPPublicKeyRing> parsedPublicKeyRings = null;
        Collection<FileItem> items = fileUploadHandler.parseItems(request, MAX_KEY_FILE_UPLOAD_SIZE);
        String keyPassword = null;
        String newPassword = null;
        PGPKeyRingParser keyParser = Services.getService(PGPKeyRingParser.class);
        if (items.size() > 0) {
            //Get actual Key data
            for (FileItem item : items) {
                if(item.getFieldName().equals(KEY_PASSWORD_PARAMETER_NAME)) {
                    keyPassword = item.getString(StandardCharsets.UTF_8.displayName());
                }
                else if(item.getFieldName().equals(NEW_PASSWORD_PARAMETER_NAME)) {
                    newPassword = item.getString(StandardCharsets.UTF_8.displayName());
                }
                else{
                    KeyRingParserResult parseResult = keyParser.parse(item.getInputStream());
                    if(parseResult != null) {
                        parsedSecretKeyRings = new ArrayList<PGPSecretKeyRing>();
                        parsedSecretKeyRings.addAll(parseResult.toSecretKeyRings());
                        parsedPublicKeyRings = new ArrayList<PGPPublicKeyRing>();
                        parsedPublicKeyRings.addAll(parseResult.toPublicKeyRings());
                    }
                }
            }
        } else {
            throw GuardCoreExceptionCodes.MULTIPART_UPLOAD_MISSING.create();
        }
        if((parsedPublicKeyRings == null || parsedPublicKeyRings.isEmpty()) &&
           (parsedSecretKeyRings == null || parsedSecretKeyRings.isEmpty()) ) {
            throw GuardCoreExceptionCodes.INVALID_PARAMETER_VALUE.create(KEY_PARAMETER_NAME);
        }
        final String email = new EmailValidator().assertInput(new Api(new OxCookie(request.getCookies()),request).getPrimary(), "email");
        //Check if password parameters are present when importing secret keys
        if(parsedSecretKeyRings.size() > 0) {
            if (keyPassword == null) {
                parsedSecretKeyRings.clear();
            }
            if(!parsedSecretKeyRings.isEmpty() && (newPassword == null || newPassword.isEmpty())) {
                throw GuardCoreExceptionCodes.PARAMETER_MISSING.create(NEW_PASSWORD_PARAMETER_NAME);
            }
        }

        //Import parsed keys
        KeyImportService keyImportService = Services.getService(KeyImportService.class);
        Collection<GuardKeys> importedKeyRings = keyImportService.importPrivateKeyRing(userSession.getGuardUserId(),
            userSession.getGuardContextId(),
            email,
            userSession.getLocale(),
            keyPassword,
            newPassword,
            parsedSecretKeyRings.toArray(new PGPSecretKeyRing[parsedSecretKeyRings.size()]));
        importedKeyRings.addAll(
            keyImportService.importPublicKeyRing(userSession.getGuardUserId(),
            userSession.getGuardContextId(),
            email,
            userSession.getLocale(),
            parsedPublicKeyRings.toArray(new PGPPublicKeyRing[parsedPublicKeyRings.size()])));

        //the result list can contain duplicates because if public + private keys are imported / merged keys are also returned.
        //Cleaning up for a client.
        importedKeyRings = removeDuplicatedImportedKeyResults(importedKeyRings);
        ServletUtils.sendObject(response, new GuardKeyInfoCollectionResponse(newPassword, importedKeyRings));
    }

}
