/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.servlets.keymanagement.actions;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseAction;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseWrapper;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.RevocationReason;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link RevokeKeyAction} marks a key ring as "revoked"
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class RevokeKeyAction extends GuardServletAction {

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        final boolean mandatory = true;
        final long keyId = ServletUtils.getLongParameter(request, Parameters.PARAM_KEYID, mandatory);
        final UserIdentity userIdentity = requireUserIdentityFrom(request, userSession.getUserId(), userSession.getContextId());
        final String reason = ServletUtils.getStringParameter(request, Parameters.PARAM_REASON);
        final GuardKeyService keyService = Services.getService(GuardKeyService.class);
        final GuardKeys key = keyService.getKeysBySubKeyId(userSession.getGuardUserId(), userSession.getGuardContextId(), keyId);
        if (key != null) {

            //Revoke the key in a brute force protected way
            AntiAbuseWrapper bruteForceProtection = getAntiAbuseWrapper(key, AntiAbuseUtils.getAllowParameters(request, userSession.getUserId(),  userSession.getContextId(), new String(userIdentity.getPassword())));
            bruteForceProtection.doAction(new AntiAbuseAction<Void>() {

                @Override
                public Void doAction() throws OXException {

                    RevocationReason revocationReason = RevocationReason.NO_REASON;
                    if (reason != null && !reason.isEmpty()) {
                        try {
                            revocationReason = RevocationReason.valueOf(reason.toUpperCase());
                        } catch (IllegalArgumentException e) {
                            throw GuardCoreExceptionCodes.INVALID_PARAMETER_VALUE.create(e, Parameters.PARAM_REASON);
                        }
                    }

                    keyService.revokeKey(key, 0, revocationReason, new String(userIdentity.getPassword()));
                    return null;
                }
            });
        } else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND.create();
        }
    }
}
