/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.milter.service.impl;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.channels.WritableByteChannel;
import java.util.ArrayList;
import org.slf4j.Logger;
import com.openexchange.exception.OXException;
import com.openexchange.guard.milter.actions.MilterEOBAction;
import com.openexchange.guard.milter.actions.MilterRcptAction;
import com.openexchange.guard.milter.actions.encryptor.MilterRcptActionEncryptor;
import com.openexchange.guard.milter.internal.MilterConstants;
import com.openexchange.guard.milter.internal.MilterPacket;
import com.openexchange.guard.milter.internal.MilterUtils;
import com.openexchange.guard.milter.service.MilterHandlerService;
import com.openexchange.guard.milter.service.MilterResponse;

/**
 * {@link GuardMilterHandlerService}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.8.0
 */
public class GuardMilterHandlerService implements MilterHandlerService {

    private static Logger Log = org.slf4j.LoggerFactory.getLogger(GuardMilterHandlerService.class);

    private GuardMilterMail mail;
    private ArrayList<MilterRcptAction> rcptActions;
    private ArrayList<MilterEOBAction> eobActions;

    public GuardMilterHandlerService () {
        super();
        mail = new GuardMilterMail();
        rcptActions = new ArrayList<MilterRcptAction>();
        eobActions = new ArrayList<MilterEOBAction> ();
    }

    @Override
    public void reset() {
        mail = new GuardMilterMail();
    }
    /* (non-Javadoc)
     * @see com.openexchange.guard.milter.service.MilterHandlerService#rcptTo(com.openexchange.guard.milter.internal.MilterPacket)
     */
    @Override
    public MilterResponse rcptTo(MilterPacket packet) throws OXException {
        String [] rcpts = MilterUtils.getStringArray(packet.getData());
        mail.addRcpt(rcpts[0]);
        boolean continueMilter = true;
        // Loop through rcptActions
        for (int i = 0; i < rcptActions.size() && continueMilter; i++) {
            continueMilter = rcptActions.get(i).handleRcpt(mail, rcpts[0]) && continueMilter;
        }
        if (!continueMilter) Log.debug("Milter Exiting, recipient not requesting encryption");
        // If we should continue, send continue, otherwise send accept
        if (continueMilter) return new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
        return new StandardResponse(MilterConstants.RESPONSE_ACCEPT);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.milter.service.MilterHandlerService#header(com.openexchange.guard.milter.internal.MilterPacket)
     */
    @Override
    public MilterResponse header(MilterPacket packet) {
        String [] headers = MilterUtils.getStringArray(packet.getData());
        mail.addHeader(headers[0], headers[1]);
        if (headers[1] !=null && headers[1].contains("application/pgp") || headers[1].contains("multipart/encrypted")) {  // abort milter if already encrypted
            return new StandardResponse(MilterConstants.RESPONSE_ACCEPT);
        }
        return new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
    }
    /* (non-Javadoc)
     * @see com.openexchange.guard.milter.service.MilterHandlerService#body(com.openexchange.guard.milter.internal.MilterPacket)
     */
    @Override
    public MilterResponse body(MilterPacket packet) {
        try {
            mail.addBody(packet.getData().array());
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
        }
        return new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
    }

    @Override
    public GuardMilterMail getMail () {
        return mail;
    }

    @Override
    public MilterResponse handleEOB(MilterPacket packet, WritableByteChannel channel) throws UnsupportedEncodingException, IOException {
        for (int i = 0; i < eobActions.size(); i++) {
            if (eobActions.get(i).handleMessageBody(mail)) {
                eobActions.get(i).sendResponses(channel);
            }
        }
        return new StandardResponse(MilterConstants.RESPONSE_CONTINUE);
    }

    @Override
    public void addRcptAction(MilterRcptAction action) {
        rcptActions.add(action);

    }

    @Override
    public void addEOBAction(MilterEOBAction action) {
        eobActions.add(action);

    }

}
