/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.keymanagement.services.impl.osgi;

import java.security.Security;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Optional;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.osgi.framework.BundleContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.contact.provider.composition.IDBasedContactsAccessFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.autocrypt.AutoCryptService;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.database.GuardShardingService;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.hkpclient.services.HKPClientService;
import com.openexchange.guard.keymanagement.keysources.CompositeKeyPairSource;
import com.openexchange.guard.keymanagement.keysources.DbCachingKeyPairSource;
import com.openexchange.guard.keymanagement.keysources.RealtimeKeyPairSource;
import com.openexchange.guard.keymanagement.services.AccountCreationService;
import com.openexchange.guard.keymanagement.services.ContactKeyService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.ImportSystemPublicKey;
import com.openexchange.guard.keymanagement.services.KeyCreationService;
import com.openexchange.guard.keymanagement.services.KeyImportService;
import com.openexchange.guard.keymanagement.services.KeyPairSource;
import com.openexchange.guard.keymanagement.services.KeyRecoveryService;
import com.openexchange.guard.keymanagement.services.MasterKeyService;
import com.openexchange.guard.keymanagement.services.PasswordChangeService;
import com.openexchange.guard.keymanagement.services.PublicExternalKeyService;
import com.openexchange.guard.keymanagement.services.RecipKeyService;
import com.openexchange.guard.keymanagement.services.impl.AccountCreationServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.ContactKeyServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.GuardKeyServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.ImportSystemPublicKeyImpl;
import com.openexchange.guard.keymanagement.services.impl.KeyCreationServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.KeyImportServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.KeyRecoveryServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.PasswordChangeServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.PublicExternalKeyServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.RecipKeyServiceImpl;
import com.openexchange.guard.keymanagement.services.impl.exceptions.GuardMasterKeyExceptionCodes;
import com.openexchange.guard.keymanagement.services.impl.masterKey.CachingMasterKeyService;
import com.openexchange.guard.keymanagement.services.impl.masterKey.IndexMasterKeyService;
import com.openexchange.guard.keymanagement.storage.KeyCacheStorage;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.keymanagement.storage.OGPGPKeysStorage;
import com.openexchange.guard.keymanagement.storage.PGPKeysStorage;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.secondfactor.SecondFactorService;
import com.openexchange.guard.session.GuardSessionService;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.guard.user.OXUserService;
import com.openexchange.guard.wksclient.WKSClientService;
import com.openexchange.osgi.HousekeepingActivator;
import com.openexchange.pgp.keys.parsing.PGPKeyRingParser;
import com.openexchange.security.manager.SecurityManagerPropertyProvider;
import com.openexchange.user.UserService;

/**
 * {@link KeyManagementServicesImplActivator}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class KeyManagementServicesImplActivator extends HousekeepingActivator {

    private static final Logger LOG = LoggerFactory.getLogger(KeyManagementServicesImplActivator.class);
    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#getNeededServices()
     */
    @Override
    protected Class<?>[] getNeededServices() {
        return new Class<?>[] { GuardCipherFactoryService.class, GuardConfigurationService.class, GuardShardingService.class, GuardRatifierService.class,
                                KeyTableStorage.class, EmailStorage.class, KeyCacheStorage.class, PGPKeysStorage.class, OGPGPKeysStorage.class,
                                OXUserService.class, GuardAntiAbuseService.class, GuardSessionService.class, GuardTranslationService.class,
            HKPClientService.class, WKSClientService.class, PGPKeyRingParser.class, IDBasedContactsAccessFactory.class, SecondFactorService.class };
    }

    @Override
    protected Class<?>[] getOptionalServices() {
        return new Class<?>[] { AutoCryptService.class };
    }

    private int getSymetricKeyAlgorithm() {
        GuardConfigurationService guardConfigurationService = getService(GuardConfigurationService.class);
        int rsaKeyLength = guardConfigurationService.getIntProperty(GuardProperty.aesKeyLength);
        switch(rsaKeyLength) {
            case 128:
                return PGPEncryptedData.AES_128;
            case 192:
                return PGPEncryptedData.AES_192;
            case 256:
                return PGPEncryptedData.AES_256;
            default:
                throw new IllegalArgumentException("Invalid configured AES keysize: " + rsaKeyLength);
        }
    }

    private int getKeyExpirationDays() {
        GuardConfigurationService guardConfigurationService = getService(GuardConfigurationService.class);
        return guardConfigurationService.getIntProperty(GuardProperty.keyValidDays);
    }

    private KeyPairSource[] getKeyPairCreationStrategies(GuardConfigurationService guardConfigurationService,
                                                         GuardCipherService guardCipherService,
                                                         KeyCacheStorage keyCacheStorage){
        Collection<KeyPairSource >strategies = new ArrayList<KeyPairSource>();
        if(guardConfigurationService.getBooleanProperty(GuardProperty.rsaCache)) {
            //Using pre generated Key data from the key cache
            strategies.add(new DbCachingKeyPairSource(this));
        }
        strategies.add(new RealtimeKeyPairSource(guardConfigurationService));
        return strategies.toArray(new KeyPairSource[strategies.size()]);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.osgi.DeferredActivator#startBundle()
     */
    @Override
    protected void startBundle() throws Exception {
        LOG.info("Starting bundle: {}", context.getBundle().getSymbolicName());

        Services.setServiceLookup(this);
        Security.addProvider(new BouncyCastleProvider());

        final GuardConfigurationService guardConfigurationService = getService(GuardConfigurationService.class);
        final GuardCipherService guardCipherService = getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        final KeyCacheStorage keyCacheStorage = getService(KeyCacheStorage.class);

        // Register key services
        GuardKeyService keyService = new GuardKeyServiceImpl();
        registerService(GuardKeyService.class, keyService);
        trackService(GuardKeyService.class);

        //register the key recovery service
        KeyRecoveryService keyRecoveryService = new KeyRecoveryServiceImpl(this);
        registerService(KeyRecoveryService.class, keyRecoveryService);
        trackService(KeyRecoveryService.class);

        //create and registering the Key creation service
        KeyCreationService keyCreationService = new KeyCreationServiceImpl(this,
            guardCipherService,
                                                                           getKeyExpirationDays(),
                                                                           getSymetricKeyAlgorithm());
        KeyPairSource keyPairSource = new CompositeKeyPairSource(getKeyPairCreationStrategies(guardConfigurationService, guardCipherService, keyCacheStorage));
        keyCreationService.setKeyPairSource(keyPairSource);
        registerService(KeyCreationService.class, keyCreationService);
        trackService(KeyCreationService.class);

        // Password change service
        PasswordChangeService passService = new PasswordChangeServiceImpl();
        registerService(PasswordChangeService.class, passService);
        trackService(PasswordChangeService.class);

        //Register the RecipKey lookup service
        registerService(RecipKeyService.class, new RecipKeyServiceImpl(keyService));
        trackService(RecipKeyService.class);

        //Service for external public key handling
        PublicExternalKeyService publicExternalKeyService = new PublicExternalKeyServiceImpl(getService(GuardRatifierService.class), getService(OGPGPKeysStorage.class));
        registerService(PublicExternalKeyService.class, publicExternalKeyService);

        //Service for getting keys per contact
        registerService(ContactKeyService.class, new ContactKeyServiceImpl(this));

        //Service for importing client created, custom keys
        registerService(KeyImportService.class, new KeyImportServiceImpl(keyRecoveryService,keyService,
                                                                         getService(KeyTableStorage.class),
                                                                         getService(PGPKeysStorage.class)));
        //Service for importing keys from command line
        registerService(ImportSystemPublicKey.class, new ImportSystemPublicKeyImpl());

        //Account creation Service
        registerService(AccountCreationService.class, new AccountCreationServiceImpl());

        // Master key service
        MasterKeyService mKeyService = new CachingMasterKeyService(new IndexMasterKeyService(this, guardCipherService), guardCipherService);
        registerService(MasterKeyService.class, mKeyService);
        trackService(MasterKeyService.class);
        // Add security manager property
        Dictionary<String, Object> dictionary = new Hashtable<String, Object>(1);
        dictionary.put(SecurityManagerPropertyProvider.PROPS_SERVICE_KEY, GuardProperty.masterKeyPath.getFQPropertyName());
        registerService(SecurityManagerPropertyProvider.class, (property) -> {
            if (GuardProperty.masterKeyPath.getFQPropertyName().contentEquals(property)) {
                return Optional.of(guardConfigurationService.getProperty(GuardProperty.masterKeyPath));
            }
            return Optional.empty();
        }, dictionary);
        try {
            mKeyService.getMasterKey(0, true);
        } catch (OXException ex) {
            if (ex.getExceptionCode().equals(GuardMasterKeyExceptionCodes.INDEXED_MASTER_PASSWORD_FILE_MISSING)) {
                LOG.error("Missing oxguardpass file in etc directory.  Guard will not function properly");
            }
            if (ex.getExceptionCode().equals(GuardMasterKeyExceptionCodes.UNABLE_TO_RETIREVE_MASTER_PASSWORD)) {
                mKeyService.createKey(0);
            }
        }

        trackService(UserService.class);
        openTrackers();
    }

    /* (non-Javadoc)
     * @see com.openexchange.osgi.DeferredActivator#stop(org.osgi.framework.BundleContext)
     */
    @Override
    public void stop(BundleContext context) throws Exception {
        org.slf4j.LoggerFactory.getLogger(KeyManagementServicesImplActivator.class).info("Stopping bundle: {}", context.getBundle().getSymbolicName());
        unregisterService(KeyCreationService.class);
        Services.setServiceLookup(null);
    }
}
