/*
 * @copyright Copyright (c) OX Software GmbH, Germany <info@open-xchange.com>
 * @license AGPL-3.0
 *
 * This code is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with OX App Suite.  If not, see <https://www.gnu.org/licenses/agpl-3.0.txt>.
 *
 * Any use of the work other than as authorized under this license or copyright law is prohibited.
 *
 */

package com.openexchange.guard.internal.authentication;

import java.util.concurrent.locks.Lock;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.api.GuardApis;
import com.openexchange.guard.exceptions.GuardExceptionCodes;
import com.openexchange.guard.internal.AbstractGuardAccess;
import com.openexchange.guard.internal.GuardApiImpl;
import com.openexchange.guard.osgi.Services;
import com.openexchange.session.Session;
import com.openexchange.sessiond.SessiondService;

/**
 * {@link AuthenticationTokenHandler} handles the Guard authentication token
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.2
 */
public class AuthenticationTokenHandler extends AbstractGuardAccess {

    private static org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(AuthenticationTokenHandler.class);
    private static final String AUTH_TOKEN_PARAMETER = "GUARD_AUTH_TOKEN";

    /**
     * Internal method to destroy the authentication token on the remote Guard service.
     *
     * @param authToken The authentication token to destroy
     * @throws OXException
     * @throws JSONException
     */
    private void destroyAuthToken(GuardAuthenticationToken authToken) throws OXException {

        GuardApiImpl guardApi = getGuardApi(PGPCORE_ENDPOINT);
        if (guardApi != null) {
            JSONObject json = new JSONObject();
            json.putSafe("session", authToken.getGuardSessionId());
            guardApi.doCallPut(GuardApis.mapFor("action", "deleteauthtoken"), json, Void.class, null);
        } else {
            logger.error("Endpoint not available " + PGPCORE_ENDPOINT);
        }
    }

    /**
     * Gets the authentication token for a given session
     *
     * @param session The session to get the authentication token for
     * @param markAsUsed Whether to mark the token as used or not
     * @return An authentication token for the given session, or null if no authentication token was attached to the session
     * @throws OXException
     */
    public GuardAuthenticationToken getForSession(Session session, boolean markAsUsed) throws OXException {
        final Lock lock = (Lock) session.getParameter(Session.PARAM_LOCK);
        lock.lock();
        try {
            if (session.containsParameter(AUTH_TOKEN_PARAMETER)) {
                GuardAuthenticationToken authToken = GuardAuthenticationToken.fromString((String) session.getParameter(AUTH_TOKEN_PARAMETER));
                if (markAsUsed && authToken != null && authToken.getMinutesValid() < 0) {
                    authToken.setUsed();
                    setForSession(session, authToken);
                }
                return authToken;
            }
            return null;
        } finally {
            lock.unlock();
        }
    }

    /**
     * Gets the authentication token for a given session and mark the token as "used"
     *
     * @param session The session to get he authentication token for
     * @return The authentication token for the given session, or null if no authentication token was attached to the session
     * @throws OXException
     */
    public GuardAuthenticationToken getForSession(Session session) throws OXException {
        return getForSession(session, true);
    }

    /**
     * Gets the authentication token for a given session and mark the token as "used"
     *
     * @param session The session to get the authentication token for
     * @return The authentication token for the given session
     * @throws OXException If no such token was found for the given session
     */
    public GuardAuthenticationToken requireForSession(Session session) throws OXException {
        GuardAuthenticationToken authToken = getForSession(session, true);
        if (authToken != null) {
            return authToken;
        }
        throw GuardExceptionCodes.MISSING_AUTH.create();
    }

    /**
     * Attaches the given authentication token for the associated session
     *
     * @param session The session
     * @param authToken The authentication token
     * @return
     * @throws OXException
     */
    public void setForSession(Session session, GuardAuthenticationToken authToken) throws OXException {
        final Lock lock = (Lock) session.getParameter(Session.PARAM_LOCK);
        lock.lock();
        try {
            session.setParameter(AUTH_TOKEN_PARAMETER, authToken.toString());
        } finally {
            lock.unlock();
        }
        if (authToken.getMinutesValid() >= 0) {  // don't push for single use
            SessiondService sessionService = Services.getService(SessiondService.class);
            sessionService.storeSession(session.getSessionID(), false);
        }
    }

    /**
     * Internal method to remove and destroy an authentication token.
     *
     * @param session The session
     * @param destroy true, in order to invalidate the authentication on the Guard Server, false to just detach it from the session.
     * @param logout If system logout.  If session being removed the hazelcast won't be notified
     * @throws OXException
     */
    private void removeForSessionInternal(Session session, boolean destroy, boolean logout) throws OXException {
        GuardAuthenticationToken authToken = null;

        final Lock lock = (Lock) session.getParameter(Session.PARAM_LOCK);
        lock.lock();
        try {
            authToken = getForSession(session, false);
            if (authToken != null) {
                session.setParameter(AUTH_TOKEN_PARAMETER, null);
            }
        } finally {
            lock.unlock();
        }

        if (destroy && authToken != null) {
            destroyAuthToken(authToken);
        }

        if (!logout) {
            if (authToken != null && authToken.getMinutesValid() >= 0) {  // don't push for single use
                SessiondService sessionService = Services.getService(SessiondService.class);
                sessionService.storeSession(session.getSessionID(), false);
            }
        }
    }

    /**
     * Removes an authentication token for the given session.
     * <p>
     * This will NOT destroy / invalidate an authentication token on the Guard server.
     *
     * @param session The session
     * @throws OXException
     */
    public void removeForSession(Session session) throws OXException {
        final boolean destroy = false;
        final boolean logout = false;
        removeForSessionInternal(session, destroy, logout);
    }

    /**
     * Removes an authentication token for the given session.
     * <p>
     * This will also destroy i.e. invalidate an authentication token on the Guard server as well.
     *
     *
     * @param session The session
     * @throws OXException
     */
    public void destroyForSession(Session session, boolean logout) throws OXException {
        final boolean destroy = true;
        removeForSessionInternal(session, destroy, logout);
    }
}
