/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.io.InputStream;
import java.util.Arrays;
import java.util.Collection;
import javax.mail.internet.MimeMessage;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.slf4j.Logger;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.inputvalidation.RangeInputValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.mime.services.MimeParser;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.fileupload.FileUploadHandler;
import com.openexchange.pgp.mail.PGPMimeService;
import com.openexchange.server.ServiceLookup;

/**
 * {@link EncryptAction} Action to encrypt an incoming email
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.4
 */
public class EncryptAction extends GuardServletAction {

    private static final Logger Log = org.slf4j.LoggerFactory.getLogger(EncryptAction.class);
    private static GuardKeyService keyservice;
    private static PGPMimeService pgpMimeService;
    private static GuardConfigurationService guardConfigService;
    private static final String MESSAGE_FIELD_NAME = "file";

    public EncryptAction(ServiceLookup services) throws OXException {
        keyservice = services.getServiceSafe(GuardKeyService.class);
        guardConfigService = services.getServiceSafe(GuardConfigurationService.class);
        pgpMimeService = services.getServiceSafe(PGPMimeService.class);
    }

    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        RangeInputValidator<Integer> positiveValidator = new RangeInputValidator<Integer>(0, Integer.MAX_VALUE);

        final int userId = positiveValidator.assertInput(ServletUtils.getIntParameter(request, "user", true), "user");
        final int contextId = positiveValidator.assertInput(ServletUtils.getIntParameter(request, "context", true), "context");
        Log.debug("encrypting mime messsage");
        GuardKeys keys = keyservice.getKeys(userId, contextId);
        if (keys == null) {
            Log.debug("No keys found for encrypting email through encrypt_mime");
            throw OXException.general("No Keys");
        }
        FileUploadHandler fileUploadHandler = new FileUploadHandler(guardConfigService, true);
        Collection<FileItem> items = fileUploadHandler.parseItems(request);
        try (InputStream messageStream = fileUploadHandler.getFileItemStreamFrom(items, MESSAGE_FIELD_NAME, true);) {
            MimeParser mimeParser = new MimeParser(messageStream);
            if (mimeParser.isPGPMime() || mimeParser.isPGPInline()) {
                Log.debug("Guard encrypt_mime exiting, already encrypted");
                throw OXException.general("Already encrypted");
            }
            MimeMessage encrypted = pgpMimeService.encrypt(mimeParser.getMessage(), Arrays.asList(keys.getPGPPublicKey()));
            encrypted.writeTo(response.getOutputStream());

        } catch (Exception e) {
            Log.error("Error encrypting incomgin email message", e);
            throw OXException.general("Error enrypting incoming message");
        }

    }
}
