/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.authentication;

import java.util.Objects;
import javax.servlet.http.HttpServletRequest;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.handler.exceptions.GuardResponseHandlerExceptionCodes;
import com.openexchange.guard.keymanagement.commons.MasterKey;
import com.openexchange.guard.keymanagement.services.MasterKeyService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.user.UserIdentity;

import static com.openexchange.java.Autoboxing.i;

/**
 * {@link UserIdentityParameterParser}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public class UserIdentityParameterParser implements UserIdentityParser {

    private static final String PARAM_PASSWORD = "password";
    private static final String PARAM_E_PASSWORD = "e_password";
    private static final String PARAM_KEY_INDEX = "keyIndex";

    private final HttpServletRequest request;

    /**
     * Initializes a new {@link UserIdentityParameterParser}.
     * @param request The request to parse the {@link UserIdentity} from
     */
    public UserIdentityParameterParser(HttpServletRequest request) {
        this.request = request;
    }

    private String decryptPassword(String e_password, Integer index, int userId, int cid) throws OXException {
        GuardCipherFactoryService cipherFactoryService = Objects.requireNonNull(Services.getService(GuardCipherFactoryService.class));
        GuardCipherService cipherService = cipherFactoryService.getCipherService(GuardCipherAlgorithm.RSA);
        MasterKeyService masterKeyService = Objects.requireNonNull(Services.getService(MasterKeyService.class));
        MasterKey mkey;
        if (index != null) {
            mkey = masterKeyService.getMasterKey(i(index), false);
        } else {
            mkey = masterKeyService.getMasterKey(userId, cid, false);
        }
        return cipherService.decrypt(e_password, masterKeyService.getDecryptedClientKey(mkey));
    }

    private String parsePassword(int userId, int cid) throws OXException {
        String password = request.getParameter(PARAM_PASSWORD);
        if(password != null) {
            return password;
        }

        String e_password = request.getParameter(PARAM_E_PASSWORD);
        if (e_password != null) {
            Integer index = null;
            if (request.getParameter(PARAM_KEY_INDEX) != null) {
                try {
                    index = Integer.valueOf(request.getParameter(PARAM_KEY_INDEX));
                } catch (@SuppressWarnings("unused") NumberFormatException e) {
                    throw GuardResponseHandlerExceptionCodes.BAD_RSA_ERROR.create();
                }
            }
            String decryptedPassword = decryptPassword(e_password, index, userId, cid);
            if (decryptedPassword == null) {
                throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
            }
            return decryptedPassword;
        }
        return null;
    }


    /* (non-Javadoc)
     * @see com.openexchange.guard.servlets.authentication.UserIdentityParser#parse()
     */
    @Override
    public UserIdentity parse(int userId, int cid) throws OXException {
        if(request.getParameterMap().size() > 0) {
            String password = parsePassword(userId, cid);
            if(password != null) {
                return new UserIdentity(/*TODO*/null, password);
            }
        }
        return null;
    }

    @Override
    public UserIdentity parse() throws OXException {
        throw OXException.mandatoryField("userid and context required");
    }
}
