/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.update.internal;

import java.util.ArrayList;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.update.KeyUpgradeTask;

/**
 * {@link GuardKeysUpdater}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class GuardKeysUpdater {

    private static final int currentVersion = 1;
    private static final Logger LOG = LoggerFactory.getLogger(GuardKeysUpdater.class);

    ArrayList<KeyUpgradeTask> upgradeServices;
    private KeyTableStorage ogKeyTableStorage;

    public GuardKeysUpdater (KeyTableStorage ogKeyTableStorage, ArrayList<KeyUpgradeTask> upgradeServices) {
        this.ogKeyTableStorage = ogKeyTableStorage;
        this.upgradeServices = upgradeServices;
    }

    public GuardKeysUpdater (KeyTableStorage ogKeyTableStorage) {
        this.ogKeyTableStorage = ogKeyTableStorage;
        this.upgradeServices = new ArrayList<KeyUpgradeTask>();
    }

    public GuardKeysUpdater addUpgradeService (KeyUpgradeTask service) {
        this.upgradeServices.add(service);
        return this;
    }

    private JsonObject getJson (String misc) {
        if (misc == null || misc.isEmpty()) return new JsonObject();
        JsonObject metaData = new JsonParser().parse(misc).getAsJsonObject();
        return metaData;
    }

    /**
     * Parse the misc data from the key to get the current upgrade version
     * @param key
     * @return
     */
    private int getCurrentVersion (GuardKeys key) {
        String misc = key.getMisc();
        JsonObject metaData = getJson (misc);
        if (!metaData.has("version")) return 0;
        return (metaData.get("version").getAsInt());
    }

    /**
     * Update the key table to reflect done
     * @param key
     */
    private void updateKeyToCurrentVersion (GuardKeys key) {
        String misc = key.getMisc();
        JsonObject metaData = getJson (misc);
        metaData.addProperty("version", currentVersion);
        try {
            ogKeyTableStorage.updateMisc(key, metaData.toString());
        } catch (OXException e) {
            LOG.error("Problem updating key version in updater ", e);
        }
    }

    /**
     * Returns the current upgrade version
     * @return
     */
    public static int getCurrentVersion () {
        return currentVersion;
    }

    /**
     * Check the key for any needed upgrades/fixes
     * Mark as fixed once done
     * @param key
     * @param password
     */
    public void checkUpgrades (GuardKeys key, String password) {
        int current = getCurrentVersion (key);
        if (current == currentVersion) return;
        for (int i = current + 1; i <= currentVersion; i++) {
            for (KeyUpgradeTask service : upgradeServices) {
                if (service.handles(i)) {
                    try {
                        service.doUpgrade(key, password);
                    } catch (Exception ex) {
                        LOG.error("Problem during update of key ", ex);
                        return; // Terminate.  Possibly can't proceed if prior updates failed
                    }
                }
            }
        }
        updateKeyToCurrentVersion (key);
    }

}
