/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.mime.services;

import java.io.IOException;
import java.io.InputStream;
import java.util.Properties;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Part;
import javax.mail.Session;
import javax.mail.internet.MimeMessage;


/**
 * {@link PGPMimeAttachmentExtractor}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class PGPMimeAttachmentExtractor {

    /**
     * Extracts an attachment from the given MIME message
     * @param mimeMessage the message to extract the attachment from
     * @param name the name (case IN-sensitive) of the attachment to extract from the MIME message
     * @return The extracted attachment, or null if no such attachment was found
     * @throws MessagingException
     * @throws IOException
     */
    public Part getAttachmentFromMessage(MimeMessage mimeMessage, String name) throws MessagingException, IOException {

    	if (mimeMessage == null) return null;
        Multipart multiPart = (Multipart)mimeMessage.getContent();

        //Iterating over all body parts and search for the part with the given name
        return parseMultipartForName (multiPart, name);

    }

    /**
     * Parse all multiparts of a message for a file name
     * @param multiPart
     * @param name
     * @return
     * @throws MessagingException
     * @throws IOException
     */
    private Part parseMultipartForName (Multipart multiPart, String name) throws MessagingException, IOException {
    	for (int i = 0; i < multiPart.getCount(); i++) {
            Part bodyPart = multiPart.getBodyPart(i);
            String type = bodyPart.getContentType();
            if (type != null && type.contains("multipart")) {
            	Part r = parseMultipartForName((Multipart) bodyPart.getContent(), name);
            	if (r != null) return (r);
            }
            String filename = bodyPart.getFileName();
            if (filename != null) {
                if (filename.toLowerCase().trim().equals(name.toLowerCase().trim())) {
                    return bodyPart;
                }
            }
        }

        return null;
    }

    /**
     * Extracts an attachment from the given InputStream containing a MIME Message
     * @param mimeMessageStream The InputStream containing the MIME Message
     * @param name the name (case IN-sensitive) of the attachment to extract from the given MIME stream
     * @return the extracted attachment, or null if no such attachment was found
     * @throws MessagingException
     * @throws IOException
     */
    public Part getAttachmentFromMessage(InputStream mimeMessageStream, String name) throws MessagingException, IOException {
        Session session = Session.getDefaultInstance(new Properties());
        return getAttachmentFromMessage(new MimeMessage(session,mimeMessageStream), name);
    }

    /**
     * Extracts an attachment from the given MIME message
     * @param mimeMessage the message to extract the attachment from
     * @param contentId The ID of the attachment to extract
     * @return The extracted attachment, or null if no such attachment was found
     * @throws MessagingException
     * @throws IOException
     */
    public Part getAttachmentFromMessageID(MimeMessage message, String contentId) throws IOException, MessagingException {

    	if (message == null) return null;
        Multipart mp = (Multipart) message.getContent();

        return parseMultipartForId(mp, contentId);
    }

    /**
     * Parse all multiparts for a contentID attachment
     * @param mp
     * @param contentId
     * @return
     * @throws IOException
     * @throws MessagingException
     */
    private Part parseMultipartForId (Multipart mp, String contentId) throws IOException, MessagingException {
    	for (int i = 0; i < mp.getCount(); i++) {
            Part p = mp.getBodyPart(i);
            String type = p.getContentType();
            if (type != null && type.contains("multipart")) {
            	Part r = parseMultipartForId ((Multipart) p.getContent(), contentId);
            	if (r != null) return (r);
            }
            String[] cids = p.getHeader("Content-ID");
            String cid = null;
            if (cids != null) {
                cid = cids[0].replace("<", "").replace(">", "");
                if (cid.toLowerCase().trim().equals(contentId.toLowerCase().trim())) {
                    return p;
                }
            }
        }
    	return null;
    }

    /**
     * Extracts an attachment from the given InputStream containing a MIME Message
     * @param mimeMessageStream The InputStream containing the MIME Message
     * @param contentId The ID of the attachment to extract
     * @return the extracted attachment, or null if no such attachment was found
     * @throws MessagingException
     * @throws IOException
     */
    public Part getAttachmentFromMessageID(InputStream mimeMessageStream, String contentId) throws MessagingException, IOException {
        Session session = Session.getDefaultInstance(new Properties());
        return getAttachmentFromMessageID(new MimeMessage(session,mimeMessageStream), contentId);
    }
}
