/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.guest.impl;

import java.io.Serializable;
import java.util.Objects;
import com.openexchange.exception.OXException;
import com.openexchange.guard.caching.GenericCache;
import com.openexchange.guard.caching.GenericCacheFactory;
import com.openexchange.guard.email.storage.ogEmail.Email;
import com.openexchange.guard.email.storage.ogEmail.EmailStorage;
import com.openexchange.guard.email.storage.ogEmail.EmailStorageListener;
import com.openexchange.guard.guest.GuestLookupService;

/**
 * {@link CachingGuestLookupService}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class CachingGuestLookupService implements GuestLookupService, EmailStorageListener {

    private final GenericCache<Email> cache;
    private final GenericCache<Integer> idCache;
    private final GuestLookupService delegate;

    public CachingGuestLookupService(GenericCacheFactory cacheFactory,
                                     GuestLookupService delegate,
                                     EmailStorage emailStorage) {
        this.cache = Objects.requireNonNull(cacheFactory, "cacheFactory must not be null").createCache("GuestLookup");
        this.idCache = cacheFactory.createCache("Guest2OXLookup");
        this.delegate = Objects.requireNonNull(delegate, "delegate must not be null");
        Objects.requireNonNull(emailStorage, "emailStorage must not be null").addListener(this);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.guest.GuestLookupService#lookupGuardGuest(int, int)
     */
    @Override
    public Email lookupGuardGuest(int userId, int contextId) throws OXException {
        final Serializable cacheKey = cache.createKey(contextId, userId);
        Email email = cache.get(cacheKey);
        if (email == null) {
            email = delegate.lookupGuardGuest(userId, contextId);
            if (email != null) {
                cache.put(cacheKey, email);
            }
        }
        return email;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.email.storage.ogEmail.EmailStorageListener#deleted(int, int)
     */
    @Override
    public void deleted(int userId, int contextId) throws OXException  {
        //clearing the whole cache; no mapping from provided guard userId/cid to Appsuite userId/cid
        cache.clear();
        idCache.clear();
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.email.storage.ogEmail.EmailStorageListener#deleted(int)
     */
    @Override
    public void deleted(int contextId) throws OXException {
        cache.clear();
        idCache.clear();
  }

    @Override
    public int lookupOxUserId(String email, int contextId) throws OXException {
        final Serializable cacheKey = idCache.createKey(email, contextId);
        Integer id = idCache.get(cacheKey);
        if (id == null) {
            id = delegate.lookupOxUserId(email, contextId);
            if (id != 0) {
                idCache.put(cacheKey, id);
            }
        }
        return id;
    }

}
