/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.pgpmail;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import javax.mail.MessagingException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadException;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.inputvalidation.RangeInputValidator;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseAction;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseWrapper;
import com.openexchange.guard.mime.services.GuardParsedMimeMessage;
import com.openexchange.guard.mime.services.MimeEncryptionService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.fileupload.FileUploadHandler;
import com.openexchange.guard.servlets.fileupload.JsonFileUploadHandler;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link DecryptMimeEmailAction}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.4
 */
public class DecryptMimeEmailAction extends GuardServletAction {

    private static final String JSON_DATA_FIELD_NAME = "json";
    private static final String JSON_AUTH_FIELD_NAME = "guardAuth";
    private static final String MESSAGE_FIELD_NAME = "file";

    private GuardParsedMimeMessage parseMimeMessage(HttpServletRequest request, JSONObject json, InputStream messageStream, int userId, int contextId) throws MessagingException, JSONException, OXException, FileUploadException, IOException {
        return ParseIncoming.parse(request, json, messageStream, userId, contextId, null, null, false);
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(final HttpServletRequest request, final HttpServletResponse response, GuardUserSession userSession) throws Exception {
        RangeInputValidator<Integer> positiveValidator = new RangeInputValidator<Integer>(0, Integer.MAX_VALUE);

        final boolean mandatory = true;
        final int userId = positiveValidator.assertInput(ServletUtils.getIntParameter(request, "user", mandatory), "user");
        final int contextId = positiveValidator.assertInput(ServletUtils.getIntParameter(request, "context", mandatory), "context");

        FileUploadHandler fileUploadHandler = new FileUploadHandler(Services.getService(GuardConfigurationService.class));
        JsonFileUploadHandler jsonFileUploadHandler = new JsonFileUploadHandler(fileUploadHandler);
        Collection<FileItem> items = fileUploadHandler.parseItems(request);
        try (InputStream messageStream = fileUploadHandler.getFileItemStreamFrom(items, MESSAGE_FIELD_NAME, true);) {

            final JSONObject json = jsonFileUploadHandler.getJsonFrom(items, JSON_DATA_FIELD_NAME, false);
            final JSONObject guardAuth = jsonFileUploadHandler.getJsonFrom(items, JSON_AUTH_FIELD_NAME, false);

            //Extracting user identity from the authenticationToken
            UserIdentity userIdentity = getUserIdentityFrom(guardAuth);
            if (userIdentity == null) {
                throw GuardAuthExceptionCodes.BAD_PASSWORD.create();
            }

            //This abuse wrapper does protect from wrong passwords inside of a valid auth-Token. Not from non-valid auth-tokens.
            AntiAbuseWrapper antiAbuseWrapper = getAntiAbuseWrapper(AntiAbuseUtils.getAllowParameters(request, userId, contextId, new String(userIdentity.getPassword())));
            final UserIdentity closureUserIdentity = userIdentity;
            antiAbuseWrapper.doAction(new AntiAbuseAction<Void>() {

                @Override
                public Void doAction() throws Exception {
                    final GuardParsedMimeMessage parsedMimeMessage = parseMimeMessage(request, json, messageStream, userId, contextId);
                    MimeEncryptionService mimeEncryptor = Services.getService(MimeEncryptionService.class);
                    mimeEncryptor.doDecryption(parsedMimeMessage, userId, contextId, closureUserIdentity, response.getOutputStream());
                    return null;
                }
            });
        }
    }

}
