/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.keymanagement.actions;

import java.nio.charset.StandardCharsets;
import java.util.Collection;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.bouncycastle.openpgp.PGPSecretKeyRing;
import com.google.gson.JsonObject;
import com.openexchange.guard.autocrypt.AutoCryptService;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.KeyImportService;
import com.openexchange.guard.keymanagement.services.PasswordChangeService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.ApiResponse;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;

/**
 * {@link ImportAutocryptKeys}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.2
 */
public class ImportAutocryptKeys extends GuardServletAction {

    private final String ID_PARAM = "id";
    private final String FOLDER_PARAM = "folder";
    private final String ATTACHMENT_PARAMT = "attachment";
    private final String PASSCODE = "startkey";
    private final String OLD_PASSWORD = "password";
    private final String NEW_PASSWORD = "newpassword";

    /* (non-Javadoc)
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request, userSession.getUserId(), userSession.getContextId());
        if (!json.has(PASSCODE) || !json.has(OLD_PASSWORD) || !json.has(NEW_PASSWORD)) {
            throw new IllegalArgumentException("Missing password");
        }
        String passcode = json.get(PASSCODE).getAsString();
        String oldpass = json.get(OLD_PASSWORD).getAsString();
        String newpass = json.get(NEW_PASSWORD).getAsString();
        Api api = new Api(new OxCookie(request.getCookies()), request);
        final String email = new EmailValidator().assertInput(api.getPrimary(), "email");
        String id = ServletUtils.getStringParameter(request, ID_PARAM, true);
        String folder = ServletUtils.getStringParameter(request, FOLDER_PARAM, true);
        String att = ServletUtils.getStringParameter(request, ATTACHMENT_PARAMT, true);
        ApiResponse resp = api.getPlainAttachment(id, att, folder);
        try {
            String content = new String(resp.readContent(), StandardCharsets.UTF_8);
            PasswordChangeService passwordService = Services.getService(PasswordChangeService.class);
            passwordService.checkPasswordRequirements(newpass);
            AutoCryptService autoCryptService = Services.getService(AutoCryptService.class);
            PGPSecretKeyRing secKeyRing = autoCryptService.getKeyFromAttachment(content, passcode);
            if (secKeyRing != null) {
                KeyImportService keyImportService = Services.getService(KeyImportService.class);
                Collection<GuardKeys> importedKeyRings = keyImportService.importPrivateKeyRing(userSession.getGuardUserId(),
                    userSession.getGuardContextId(),
                    email,
                    userSession.getLocale(),
                    oldpass,
                    newpass,
                    secKeyRing);
                if (importedKeyRings != null && importedKeyRings.size() > 0) {
                    ServletUtils.sendObject(response, "Success");
                    return;
                }
            }
        } finally {
            resp.close();
        }
        ServletUtils.sendError(response, "Unable to import key");

    }

}
