/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.keymanagement.actions;

import static com.openexchange.guard.servlets.keymanagement.actions.Parameters.PARAM_EMAIL;
import static com.openexchange.guard.servlets.keymanagement.actions.Parameters.PARAM_NAME;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.KeyCreationService;
import com.openexchange.guard.keymanagement.services.KeyRecoveryService;
import com.openexchange.guard.keymanagement.services.MasterKeyService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.servlets.GuardServletAction;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.keymanagement.responses.GuardKeyInfoResponse;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link CreateKeyPairAction} creates a new key pair, i.e. a new {@link GuardKeys} object, for a user.
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class CreateKeyPairAction extends GuardServletAction {

    private GuardKeys createKey(GuardUserSession userSession, HttpServletRequest request, UserIdentity userIdentity) throws OXException {
        KeyCreationService keyCreationService = Services.getService(KeyCreationService.class);
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        KeyRecoveryService keyRecoveryService = Services.getService(KeyRecoveryService.class);

        final String name = ServletUtils.getStringParameter(request, PARAM_NAME, true);
        final String secondaryEmail = ServletUtils.getStringParameter(request, PARAM_EMAIL, false);
        final String email = new EmailValidator().assertInput(new Api(new OxCookie(request.getCookies()), request).getPrimary());
        final boolean createRecovery = keyRecoveryService.checkCreateRecovery(userSession.getContextId(), userSession.getUserId());
        GuardKeys existingKey = keyService.getKeys(userSession.getGuardUserId(), userSession.getGuardContextId());
        final boolean isFirstKey = existingKey == null;
        final boolean markAsCurrent = true;
        boolean hasSecondary = false;

        if (secondaryEmail != null && !secondaryEmail.isEmpty()) {
            new EmailValidator().assertInput(secondaryEmail);
            hasSecondary = true;
        }

        GuardKeys keys = keyCreationService.create(userSession.getGuardUserId(),
            userSession.getGuardContextId(),
            name,
            email,
            new String(userIdentity.getPassword()),
            userSession.getLocale(),
            markAsCurrent,
            createRecovery,
            userSession.getUserId(),
            userSession.getContextId());

        if (isFirstKey) {
            final boolean isUserCreatedKey = true;
            keys = keyService.storeKeys(keys, isUserCreatedKey);
        } else {
            keyService.addNewKey(keys);
            if(existingKey != null) {
                keyService.updateSettingsForUser(existingKey);
                keys.setQuestion(existingKey.getQuestion());
                MasterKeyService mKeyService = Services.getService(MasterKeyService.class);
                if (mKeyService != null) {
                    String decrypted = mKeyService.getRcDecrypted(existingKey.getAnswer(), existingKey.getQuestion(), existingKey.getMasterKeyIndex());
                    keys.setAnswer(mKeyService.getRcEncryted(decrypted, keys.getQuestion(), keys.getMasterKeyIndex()));
                }
                keyService.updateAnswerQuestionForUser(keys);
            }
        }

        if (hasSecondary) {
            keyService.storeQuestion(keys.getUserid(), keys.getContextid(), "e", secondaryEmail);
        }

        return keys;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.servlets.GuardServletAction#doActionInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse, com.openexchange.guard.servlets.authentication.GuardUserSession)
     */
    @Override
    protected void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception {
        if (userSession == null) {
            throw GuardAuthExceptionCodes.MISSING_SESSION.create();
        }

        final UserIdentity userIdentity = requireUserIdentityFrom(request, userSession.getUserId(), userSession.getContextId());
        final GuardKeys keys = createKey(userSession, request, userIdentity);

        ServletUtils.sendObject(response, new GuardKeyInfoResponse(keys, new String(userIdentity.getPassword())));
    }

}
