/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.guestreader;

import java.io.IOException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.http.HttpStatus;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.osgi.Services;


/**
 * {@link GuestReaderServlet} - Redirecting GET requests to Guard's external guest reader  
 * <p>
 *   With OX Guard 2.4.0 the external guest reader is located in it's own package and not served by the servlet container anymore.
 *   We provide a redirect to the new location of the guest reader in order to access old guest-reader links.
 * </p> 
 * 
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public class GuestReaderServlet extends HttpServlet {
    
    private static final long serialVersionUID = 3165090161928784444L;
    private static final Logger logger = LoggerFactory.getLogger(GuestReaderServlet.class);
    private static final String FALLBACK_DEFAUL_READER_PATH = "/guard/reader/reader.html";
    private static final String LOCATION_HEADER = "Location";

    /**
     * Internal method read the location to redirect the incoming request
     * @param req the incoming request
     * @return the location to redirect the request
     * @throws OXException
     */
    private String getNewLocation(HttpServletRequest req) throws OXException {
        GuardConfigurationService configurationService = Services.getService(GuardConfigurationService.class);
        String readerPath = configurationService.getProperty(GuardProperty.externalReaderPath);
        //remove the domain part from the path
        if(readerPath.contains("/")) {
            readerPath = readerPath.substring(readerPath.indexOf("/"));
        }
        return readerPath;
    }
    
    /**
     * Internal method to read the query string from the incoming request
     * @param req the incoming request
     * @return the query string from the incoming request starting with "?", or and empty string, if the request does not contain a query string
     */
    private String getQueryString(HttpServletRequest req) {
        String queryString = req.getQueryString();
        return (queryString != null && !queryString.isEmpty()) ? "?" + queryString : "";
    }

    /**
     * Internal method to perform the redirect
     * @param req the incoming request
     * @param resp the response
     * @throws IOException
     */
    private void doRedirect(HttpServletRequest req, HttpServletResponse resp) throws IOException {
        try {            
            String newLocation = getNewLocation(req) + getQueryString(req);
            logger.debug(String.format("Redirecting OX Guard reader request to %s", newLocation));
            resp.setStatus(HttpStatus.SC_MOVED_PERMANENTLY);
            resp.setHeader(LOCATION_HEADER, newLocation);
        }
        catch(Exception e) {
            logger.error("Error while redirecting to the guest reader location", e);
            String fallback = FALLBACK_DEFAUL_READER_PATH + getQueryString(req);
            logger.debug(String.format("Redirecting OX Guard reader request to the fallback location of %s", FALLBACK_DEFAUL_READER_PATH));
            resp.setStatus(HttpStatus.SC_MOVED_PERMANENTLY);
            resp.setHeader(LOCATION_HEADER, fallback);
        }
    }
    
    /* (non-Javadoc)
     * @see javax.servlet.http.HttpServlet#doGet(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void doGet(HttpServletRequest req, HttpServletResponse resp) throws ServletException, IOException {        
        doRedirect(req, resp);
    }
}
