/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets.authentication;

import java.util.Objects;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.pgpcore.services.TokenAuthenticationService;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link UserIdentityJsonParser}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.8.3
 */
public class UserIdentityJsonParser implements UserIdentityParser {

    private static final Logger logger = LoggerFactory.getLogger(UserIdentityJsonParser.class);
    private final JSONObject json;
    private final TokenAuthenticationService tokenAuthenticationService;


    /**
     * Initializes a new {@link UserIdentityJsonParser}.
     * @param json The json to parse the auth-token from
     */
    public UserIdentityJsonParser(TokenAuthenticationService tokenAuthenticationService, JSONObject json) {
        this.tokenAuthenticationService = Objects.requireNonNull(tokenAuthenticationService, "The provided service must not be null.");
        this.json = Objects.requireNonNull(json, "The provided JSON object must not be null");
    }

    /**
     * Constructs an UserIdentity from a given authToken
     *
     * @param sessionIdentifier The client identifier
     * @param authToken The authToken
     * @return The decrypted UserIdentifier
     * @throws OXException
     */
    private UserIdentity getIdentityForToken(String sessionIdentifier, String authToken) throws OXException {
        return tokenAuthenticationService.decryptUserIdentity(sessionIdentifier, authToken);
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.servlets.pgpcore.UserIdentityFactory#create()
     */
    @Override
    public UserIdentity parse() throws OXException {
        try {
            if (json.has("user")) {
                JSONObject signer = json.getJSONObject("user");
                if (signer.has("identity") && signer.has("password")) {
                    return new UserIdentity(signer.get("identity").toString(), signer.get("password").toString());
                } else if (signer.has("session") && signer.has("auth")) {
                    UserIdentity ret = getIdentityForToken(signer.get("session").toString(), signer.get("auth").toString());
                    if (ret == null) {
                        logger.error("Unable to decode user identity from auth-token");
                    }
                    return ret;
                }
            }
            return null;
        } catch (Exception e) {
            throw GuardCoreExceptionCodes.JSON_ERROR.create(e.getMessage());
        }
    }

    @Override
    public UserIdentity parse(int userId, int cid) throws OXException {
        return parse();
    }
}
