/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.servlets;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.json.JSONObject;
import org.slf4j.event.Level;
import com.openexchange.antiabuse.AllowParameters;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.auth.AuthUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardAuthExceptionCodes;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.antiabuse.AntiAbuseWrapper;
import com.openexchange.guard.keymanagement.commons.antiabuse.KeyAntiAbuseWrapper;
import com.openexchange.guard.keymanagement.commons.antiabuse.UserAntiAbuseWrapper;
import com.openexchange.guard.logging.GuardLogProperties;
import com.openexchange.guard.logging.LoggingHelper;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.pgpcore.services.TokenAuthenticationService;
import com.openexchange.guard.servlets.authentication.AuthenticationDataExtractor;
import com.openexchange.guard.servlets.authentication.GuardAuthenticationHandler;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.servlets.authentication.GuestSessionValidator;
import com.openexchange.guard.servlets.authentication.RestSessionValidator;
import com.openexchange.guard.servlets.authentication.UserIdentityJsonParser;
import com.openexchange.guard.servlets.authentication.UserIdentityParameterParser;
import com.openexchange.guard.servlets.fileupload.AbstractTrackingFileItemFactory;
import com.openexchange.guard.servlets.fileupload.FileUploadHandler;
import com.openexchange.guard.translation.GuardTranslationService;
import com.openexchange.guard.translation.GuardTranslator;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.i18n.Translator;

/**
 * Represents an OX Guard HTTP action
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.0
 */
public abstract class GuardServletAction {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(GuardServletAction.class);

    private static final String WWW_AUTHENTICATE_HEADER = "WWW-Authenticate";
    private List<GuardAuthenticationHandler> authenticationHandlers;
    private GuardErrorResponseRenderer errorResponseRenderer;

    /**
     * Initializes a new {@link GuardServletAction}.
     */
    public GuardServletAction() {}

    /**
     * Initializes a new {@link GuardServletAction}.
     *
     * @param authenticationHandlers a set of authentication handlers
     */
    public GuardServletAction(GuardAuthenticationHandler... authenticationHandlers) {
        this.authenticationHandlers = new ArrayList<GuardAuthenticationHandler>(Arrays.asList(authenticationHandlers));
    }

    /**
     * Internal method to null-check the given {@link UserIdentity}.
     *
     * @param userIdentity The {@link UserIdentity} to check.
     * @param require true, if the method should throw an exception if the given userIdentity is null, false to return null.
     * @return The given userIdentity
     * @throws OXException
     */
    protected UserIdentity requireOrReturn(UserIdentity userIdentity, boolean require) throws OXException {
        if (userIdentity == null && require == true) {
            throw GuardAuthExceptionCodes.AUTH_MISSING.create();
        }
        return userIdentity;
    }

    /**
     * Gets the {@link org.slf4j.event.Level} to use for logging the execution of the {@link GuardServletAction}
     *
     * @return The {@link org.slf4j.event.Level} to use
     */
    protected Level getLogLevel() {
        return Level.INFO;
    }

    /**
     * Fluent like method to set the authentication handlers
     *
     * @param authenticationHandlers Authentication handlers to set
     * @return this, with the new authentication handlers set
     */
    public GuardServletAction setAuthenticationHandler(GuardAuthenticationHandler... authenticationHandlers) {
        this.authenticationHandlers = new ArrayList<GuardAuthenticationHandler>(Arrays.asList(authenticationHandlers));
        return this;
    }

    /**
     * Fluent like method to set the {@link GuardErrorResponseRenderer} which will be used for handling exceptions
     * occurred during the execution of the action.
     *
     * @param errorResponseRenderer The {@link GuardErrorResponseRenderer} used to render errors for clients.
     * @return this, with the new response renderer set.
     */
    public GuardServletAction setErrorResponseRenderer(GuardErrorResponseRenderer errorResponseRenderer) {
        this.errorResponseRenderer = errorResponseRenderer;
        return this;
    }

    /**
     * Gets the {@link GuardErrorResponseRenderer} which should be used to render exceptions for clients.
     *
     * @return The {@link GuardErrorResponseRenderer} which should be used for exception handling.
     */
    public GuardErrorResponseRenderer getErrorResponseRenderer() {
        return this.errorResponseRenderer;
    }

    /**
     * Checks authentication using all known authentication handler
     *
     * @param request the request to check authentication for
     * @param oxUserSession The user session or null if the action is not performed in a user context
     * @param response the response, used to set WWW_AUTHENTICATE_HEADER in case the authentication failed
     * @return true if authentication has been granted by all known authentication handlers of the action, false otherwise
     * @throws Exception
     */
    private boolean doAuthentication(HttpServletRequest request, GuardUserSession oxUserSession, HttpServletResponse response) throws Exception {
        if (authenticationHandlers != null && authenticationHandlers.size() > 0) {
            for (GuardAuthenticationHandler handler : authenticationHandlers) {
                if (!handler.authenticate(this, oxUserSession, request)) {
                    //Authentication failed - Adding the www-authenticate-header telling the client how authentication should be done
                    response.addHeader(WWW_AUTHENTICATE_HEADER, handler.getWWWAuthenticateHeader());
                    return false;
                }
            }
        }
        return true;
    }

    private GuardUserSession getSession(HttpServletRequest request) throws OXException {
        GuardUserSession guardUserSession = new GuestSessionValidator().validateSession(request);
        if (guardUserSession == null) {
            guardUserSession = new RestSessionValidator().validateSession(request);
            logAction(request, guardUserSession);
        } else {
            logGuestAction(request, guardUserSession);
        }
        return guardUserSession;
    }

    /**
     * Cleanup any temporary files created handling the request
     * cleanup
     *
     * @param request
     */
    private void cleanup(HttpServletRequest request) {
        Object fileItemFactory = request.getServletContext().getAttribute(FileUploadHandler.REQUEST_TRACKER);
        if (fileItemFactory == null) {
            return;
        }
        if (fileItemFactory instanceof AbstractTrackingFileItemFactory) {
            ((AbstractTrackingFileItemFactory) fileItemFactory).cleanup();
        }
    }

    /**
     * Performs the action after checking the authentication
     *
     * @param request the request related to the action
     * @param response the response related to the action
     * @throws Exception
     */
    public void doAction(HttpServletRequest request, HttpServletResponse response) throws Exception {
        //Validate session if provided by the client and get corresponding context-id and user-id
        GuardUserSession guardUserSession = getSession(request);
        //authenticate
        if (doAuthentication(request, guardUserSession, response)) {
            doActionInternal(request, response, guardUserSession);
        } else {
            if (guardUserSession == null) {  // If authentication failed, and no user session
                GuardConfigurationService guardConfig = Services.getService(GuardConfigurationService.class);
                Thread.sleep(guardConfig.getIntProperty(GuardProperty.sessionSyncDelay), 3000);
                guardUserSession = getSession(request);  // Try again.  Hazelcast delay or sync issues
                if (doAuthentication(request, guardUserSession, response)) {
                    LOG.warn("Validated session only after delay.  Verify routing from Guard to middleware");
                    doActionInternal(request, response, guardUserSession);
                    return;
                }
            }
            ServletUtils.sendResponse(response, HttpServletResponse.SC_UNAUTHORIZED, "Unauthorized");
        }
    }

    /**
     * Sets and logs current user and delegates to desired action
     *
     * @param request the request related to the action
     * @param userSession a user session or null, if invoked in a non-user context
     * @throws Exception
     */
    private void logGuestAction(HttpServletRequest request, GuardUserSession userSession) {
        if (userSession == null) {
            return;
        }
        GuardLogProperties.put(GuardLogProperties.Name.GUEST_CONTEXT_ID, userSession.getContextId());
        GuardLogProperties.put(GuardLogProperties.Name.GUEST_USER_ID, userSession.getUserId());
        GuardLogProperties.put(GuardLogProperties.Name.GUEST_SESSION_ID, userSession.getSessionId());

        LoggingHelper.Log(LOG, getLogLevel(), String.format("Command sent \"%s\" from IP %s", ServletUtils.getStringParameter(request, "action"), ServletUtils.getClientIP(request)));
    }

    /**
     * Sets and logs current user and delegates to desired action
     *
     * @param request the request related to the action
     * @param userSession a user session or null, if invoked in a non-user context
     * @throws Exception
     */
    private void logAction(HttpServletRequest request, GuardUserSession userSession) {
        if (userSession != null) {
            GuardLogProperties.put(GuardLogProperties.Name.SESSION_CONTEXT_ID, userSession.getContextId());
            GuardLogProperties.put(GuardLogProperties.Name.SESSION_USER_ID, userSession.getUserId());
            GuardLogProperties.put(GuardLogProperties.Name.SESSION_SESSION_ID, userSession.getSessionId());
        }

        LoggingHelper.Log(LOG, getLogLevel(), String.format("Command sent \"%s\" from IP %s", ServletUtils.getStringParameter(request, "action"), ServletUtils.getClientIP(request)));
    }

    /**
     * Gets the password from a given request.
     *
     * This will try to get the password from parameter, from authentication-token, or from epassword parameter
     *
     * @param request The request to get the password from
     * @return The password from the given request, or null if no password was found
     * @throws OXException
     */
    protected String getPassword(HttpServletRequest request, GuardUserSession userSession) throws OXException {
        return new AuthenticationDataExtractor().getPassword(request, userSession);
    }

    /**
     * Gets the password from a given request.
     *
     * This will try to get the password from parameter, from authentication-token, or from epassword parameter
     *
     * @param request The request to get the password from
     * @return The password from the given request, or null if no password was found
     * @throws OXException
     */
    protected String requirePassword(HttpServletRequest request, GuardUserSession userSession) throws OXException {
        String password = getPassword(request, userSession);
        if (password == null) {
            throw GuardCoreExceptionCodes.PARAMETER_MISSING.create("password or auth");
        }
        return password;
    }

    /**
     * Gets a helper/simplified wrapper around {@link GuardAntiAbuseService} for handling failed login attempts and lockouts.
     *
     * @param keys The keys
     * @param password The password of the keys
     * @param userSession The user session
     * @return The helper for handling failed login attempts and lockouts.
     * @throws OXException
     */
    protected AntiAbuseWrapper getAntiAbuseWrapper(GuardKeys keys, AllowParameters allowParams) throws OXException {
        GuardAntiAbuseService antiAbuse = Services.getService(GuardAntiAbuseService.class);
        return new KeyAntiAbuseWrapper(antiAbuse,
            keys,
            allowParams);
    }

    /**
     * Gets a helper/simplified wrapper around {@link GuardAntiAbuseService} for handling failed login attempts and lockouts.
     *
     * @param userId The user's id
     * @param contextId The context id
     * @param Password the password
     * @param remoteIP the user's IP
     * @return The helper for handling failed login attempts and lockouts.
     * @throws OXException
     */
    protected AntiAbuseWrapper getAntiAbuseWrapper(AllowParameters allowParams) throws OXException {
        GuardAntiAbuseService antiAbuse = Services.getService(GuardAntiAbuseService.class);
        return new UserAntiAbuseWrapper(antiAbuse,
            allowParams);
    }

    /**
     * Parses a {@link UserIdentity} from the given {@link HttpServletRequest}
     *
     * @param request The request to parse the {@link UserIdentity} from.
     * @return The parsed {@link UserIdentity}, or null if the request did not contain enough information for creating a {@link UserIdentity} object.
     * @throws OXException
     */
    protected UserIdentity getUserIdentityFrom(HttpServletRequest request, int userId, int cid) throws OXException {
        return new UserIdentityParameterParser(request).parse(userId, cid);
    }

    /**
     * Parses a {@link UserIdentity} from the given {@link HttpServletRequest}, or throws an Exception if the request does not contain enough information for creating a {@link UserIdentity} object.
     *
     * @param request The request to parse the {@link UserIdentity} from.
     * @return The parsed {@link UserIdentity}
     * @throws OXException
     */
    protected UserIdentity requireUserIdentityFrom(HttpServletRequest request, int userId, int cid) throws OXException {
        final UserIdentity userIdentity = getUserIdentityFrom(request, userId, cid);
        final boolean require = true;
        return requireOrReturn(userIdentity, require);
    }

    /**
     * Parses a {@link UserIdentity} from the given {@link JSONObject}
     *
     * @param json The JSON object to parse the {@link UserIdentity} from.
     * @return The parsed {@link UserIdentity}, or null if the JSON object did not contain enough information for creating a {@link UserIdentity} object.
     * @throws OXException
     */
    protected UserIdentity getUserIdentityFrom(JSONObject json) throws OXException {
        return new UserIdentityJsonParser(Services.getService(TokenAuthenticationService.class), json).parse();
    }

    /**
     * Parses a {@link UserIdentity} from the given {@link JSONObject}, or throws an Exception if the request does not contain enough information for creating a {@link UserIdentity} object.
     *
     * @param json The JSON object to parse the {@link UserIdentity} from.
     * @return The parsed {@link UserIdentity}.
     * @throws OXException
     */
    protected UserIdentity requireUserIdentityFrom(JSONObject json) throws OXException {
        final UserIdentity userIdentity = new UserIdentityJsonParser(Services.getService(TokenAuthenticationService.class), json).parse();
        final boolean require = true;
        return requireOrReturn(userIdentity, require);
    }

    /**
     * Parses a {@link UserIdentity} from the given {@link GuardUserSession} and the provided authentication token
     *
     * @param guardSession The user session to use
     * @param authToken The auth token to use
     * @return The parsed {@link UserIdentity}, or null if the request did not contain enough information for creating a {@link UserIdentity} object.
     * @throws OXException
     */
    protected UserIdentity getUserIdentityFrom(GuardUserSession guardSession, String authToken) throws OXException {
        TokenAuthenticationService service = Services.getService(TokenAuthenticationService.class);
        return service.decryptUserIdentity(guardSession.getGuardSession(), authToken);
    }

    /**
     * Parses a {@link UserIdentity} from the given {@link GuardUserSession} and the provided authentication token, or throws an Exception if the request does not contain enough information for creating a {@link UserIdentity} object.
     *
     * @param guardSession The user session to use
     * @param authToken The auth token to use
     * @return The parsed {@link UserIdentity}, or null if the request did not contain enough information for creating a {@link UserIdentity} object.
     * @throws OXException
     */
    protected UserIdentity requireUserIdentityFrom(GuardUserSession guardSession, String authToken) throws OXException {
        final UserIdentity userIdentity = getUserIdentityFrom(guardSession, authToken);
        final boolean require = true;
        return requireOrReturn(userIdentity, require);
    }

    /**
     * Gets a {@link UserIdentity} from the given {@link GuardUserSession}
     *
     * @param guardUserSession The user session
     * @return The {@link UserIdentity} from the given {@link GuardUserSession}
     * @throws OXException
     */
    protected UserIdentity getUserIdentityFrom(GuardUserSession guardUserSession) throws OXException {
        return AuthUtils.getUserIdentityFrom(guardUserSession);
    }

    /**
     * Gets a {@link Translator} for the given session's locale
     *
     * @param guardUserSession The session containing the locale
     * @return The translator for the given sesison's locale, or the default translator
     * @throws OXException
     */
    protected GuardTranslator getTranslatorFrom(GuardUserSession guardUserSession) throws OXException {
        return new GuardTranslator(Services.getService(GuardTranslationService.class), guardUserSession.getLocale());
    }

    /**
     * Template method which allows implementation of the action execution by subclasses
     *
     * @param request the request related to the action
     * @param response the response related to the action
     * @param userSession a user session or null, if invoked in a non-user context
     * @throws Exception
     */
    protected abstract void doActionInternal(HttpServletRequest request, HttpServletResponse response, GuardUserSession userSession) throws Exception;
}
