/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.auth;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.common.servlets.utils.AntiAbuseUtils;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.common.util.JsonUtil;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.handler.ResponseHandler;
import com.openexchange.guard.inputvalidation.EmailValidator;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.util.PGPUtil;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.MasterKeyService;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.OxCookie;

/**
 * {@link SecondaryEmailHandler}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class SecondaryEmailHandler {

    private static Logger LOG = LoggerFactory.getLogger(SecondaryEmailHandler.class);

    /**
     * Update secondary email address after verifying OxGuard password
     *
     * @param request
     * @param response
     * @param cookie
     * @throws Exception
     */
    public void changeSecondaryEmail(HttpServletRequest request, HttpServletResponse response, int userId, int cid, OxCookie cookie) throws Exception {
        JsonObject json = Services.getService(ResponseHandler.class).getJsonAndDecodeEncryptedPasswords(request, userId, cid);
        GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
        Api ap = new Api(cookie, request);
        if (ap.verifyLogin() == false) {// verify logged into UI
            ServletUtils.sendNotAcceptable(response, "Not logged in");
            return;
        }
        String password = JsonUtil.getStringFromJson(json, "password", true);
        final String newemail = new EmailValidator().assertInput(JsonUtil.getStringFromJson(json, "email", true), "email");
        GuardKeyService keyService = Services.getService(GuardKeyService.class);

        if (antiabuse.blockLogin(AntiAbuseUtils.getAllowParameters(request, userId, cid, password))) {
            ServletUtils.sendNotAcceptable(response, "Lockout");
            return;
        }
        GuardKeys keys = keyService.getKeys(userId, cid);
        if(keys != null) {
            if (PGPUtil.verifyPassword(keys.getPGPSecretKeyRing(), password, keys.getSalt())) {
                if (updateEmail(userId, cid, newemail)) {
                    JSONObject resp = new JSONObject();
                    resp.put("data", "OK");
                    ServletUtils.sendHtmlOK(response, resp.toString());
                    antiabuse.report(AntiAbuseUtils.getReportParameter(true, request, userId,  cid, password));
                } else {
                    LOG.info("Problem updating secondary email");
                    ServletUtils.sendNotAcceptable(response, "Fail");
                }
            } else {
                antiabuse.report(AntiAbuseUtils.getReportParameter(false, request, userId,  cid, password));
                LOG.info("Bad password for changing secondary email");
                ServletUtils.sendNotAcceptable(response, "Bad password");
                return;
            }
        }
        else {
            throw GuardCoreExceptionCodes.KEY_NOT_FOUND_FOR_IDS_ERROR.create(userId, cid);
        }
    }

    public String getSecondaryEmail(int userid, int cid) throws Exception {
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        GuardKeys userKeys = ogKeyTableStorage.getCurrentKeyForUser(userid, cid);
        String encr_answer = userKeys.getAnswer();
        if (encr_answer == null) {
            return null;
        }
        return Services.getService(MasterKeyService.class).getRcDecrypted(encr_answer, "e", userKeys.getMasterKeyIndex());

    }

    private boolean updateEmail(int userid, int cid, String email) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        return (keyService.storeQuestion(userid, cid, "e", email));
    }


}
