/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.auth;

import java.util.List;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.JsonObject;
import com.openexchange.exception.OXException;
import com.openexchange.guard.antiabuse.GuardAntiAbuseService;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.servlets.utils.ServletUtils;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.MasterKeyService;
import com.openexchange.guard.keymanagement.storage.KeyTableStorage;
import com.openexchange.guard.mailcreator.MailCreatorService;
import com.openexchange.guard.notification.GuardNotificationService;
import com.openexchange.guard.osgi.Services;
import com.openexchange.guard.oxapi.OxCookie;
import com.openexchange.guard.oxuser.GetUserService;
import com.openexchange.guard.oxuser.GetUserServiceImpl;
import com.openexchange.guard.oxuser.OxUserResult;
import com.openexchange.guard.ratifier.GuardRatifierService;
import com.openexchange.guard.servlets.authentication.GuardUserSession;
import com.openexchange.guard.user.UserIdentity;

/**
 * {@link ResetPasswordHandler}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class ResetPasswordHandler {
    private static Logger LOG = LoggerFactory.getLogger(ResetPasswordHandler.class);


    /**
     * Handle the web request for reset
     * @param userid
     * @param cid
     * @param request
     * @param response
     * @param cookie
     * @param web
     * @throws Exception
     */
    public void resetPass(GuardUserSession userSession, HttpServletRequest request, HttpServletResponse response, OxCookie cookie, boolean web) throws Exception {
        int templid = 0;
        int userid = userSession.getUserId();
        int cid = userSession.getContextId();
        try {
            GuardConfigurationService guardConfigService = Services.getService(GuardConfigurationService.class);
            templid = guardConfigService.getIntProperty(GuardProperty.templateID, userid, cid);
        } catch (Exception e) {
            LOG.error("problem getting template id for reset password email");
        }
        KeyTableStorage ogKeyTableStorage = Services.getService(KeyTableStorage.class);
        GuardConfigurationService configService = Services.getService(GuardConfigurationService.class);
        String lang = request == null ? configService.getProperty(GuardProperty.defaultLanguage, userid, cid) : request.getParameter("lang");
        List<GuardKeys> keysForUser = ogKeyTableStorage.getKeysForUser(userid, cid);
        GuardKeys key = null;

        if (keysForUser.size() > 0) {
            key = keysForUser.get(0);
        }
        if (key == null && request != null) {
            GetUserService getUser = new GetUserServiceImpl(cookie, request);
            OxUserResult result = getUser.getUser(userid);
            if (result != null && result.isGuest()) {
                GuardKeyService keyservice = Services.getService(GuardKeyService.class);
                key = keyservice.getKeys(result.getEmail());
            }

        }
        if (key != null) {
            String encr_answer = key.getAnswer();
            String userEmail = key.getEmail();
            boolean secondary = true;
            GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
            // Mailing to secondary email address if web.  If from command line, then email to primary
            String rc = Services.getService(MasterKeyService.class).getMasterKey(key.getMasterKeyIndex(), false).getRC();
            String secondEmail;
            if (web) {
                secondEmail = (encr_answer == null) ? null: cipherService.decrypt(encr_answer, rc, "e");
            } else {
                secondEmail = userEmail;  // For reset from command line, use primary email address
            }
            if ((secondEmail == null) || secondEmail.equals("") || secondEmail.equals(userEmail)) {
                secondEmail = userEmail;
                secondary = false;
            }
            GuardRatifierService validatorService = Services.getService(GuardRatifierService.class);
            validatorService.validate(secondEmail);
            String hostname = (request == null ? "" : request.getServerName());
            String senderIp = (request == null ? null: request.getRemoteAddr());
            UserIdentity userIdentity = AuthUtils.getUserIdentityFrom(userSession);
            reset(userIdentity, response, web, templid, lang, userEmail, secondEmail, hostname, senderIp);

            if (web) {
                GuardAntiAbuseService antiabuse = Services.getService(GuardAntiAbuseService.class);
                antiabuse.removeBad(userSession.getGuardSession());
                antiabuse.removeBad(userid + "-" + cid);
                if (secondary) {
                    ServletUtils.sendHtmlOK(response, "ok");
                } else {
                    ServletUtils.sendHtmlOK(response, "primary");
                }
            } else {
                System.out.println("OK");
            }
        } else {// Bad secondary email
            if (web) {
                ServletUtils.sendHtmlOK(response, "NoSecondary");
            } else {
                System.out.println("No Secondary Email");
            }
            return;
        }
    }

    /**
     * Performs the actual reset
     * @param userid
     * @param cid
     * @param response
     * @param web
     * @param templid
     * @param lang
     * @param userEmail
     * @param secondEmail
     * @param host
     * @throws OXException
     */
    public void reset(UserIdentity userIdentity, HttpServletResponse response, boolean web, int templid, String lang, String userEmail, String secondEmail, String host, String senderIp) throws OXException {
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        boolean recoveryEnabled = keyService.isRecoveryEnabled(
            userIdentity.getOXGuardUser().getId(),
            userIdentity.getOXGuardUser().getContextId());
        if (!recoveryEnabled) {
            if (web) {
                LOG.info("Unable to recover password due to no recovery");
                ServletUtils.sendHtmlOK(response, "NoRecovery");
            } else {
                System.out.println("No recovery available");
            }
            throw GuardCoreExceptionCodes.DISABLED_ERROR.create("reset password");
        }
        String newpass = keyService.generatePassword(userIdentity.getOXUser().getId(), userIdentity.getOXUser().getContextId());

        MailCreatorService mailCreatorService = Services.getService(MailCreatorService.class);
        GuardNotificationService guardNotificationService = Services.getService(GuardNotificationService.class);
        int senderId = userIdentity.getOXUser() == null ? -1 : userIdentity.getOXUser().getId();
        int senderCid = userIdentity.getOXUser() == null ? userIdentity.getOXGuardUser().getContextId() : userIdentity.getOXUser().getContextId();
        List<String> fromEmail = mailCreatorService.getFromAddress(userEmail,
            userEmail,
            senderId,
            senderCid);
        JsonObject mail = mailCreatorService.getResetEmail(
            secondEmail,
            fromEmail,
            newpass,
            lang,
            templid,
            host,
            senderId,
            senderCid);
        guardNotificationService.send(mail,
            senderId,
            senderCid,
            senderIp);
        keyService.resetPassword(userEmail, newpass);
    }


}
