/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.wksclient;

import java.util.Collection;
import java.util.Optional;
import com.openexchange.exception.OXException;

/**
 * {@link WKSClientService}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.10.2
 */
public interface WKSClientService {

    /**
     * Tries to find all (including expired and revoked) public-key rings for the given email address by performing a WKS lookup.
     * <br>
     * <br>
     * The query is performed on the host obtained from the domain part of the given email address.
     * <br>
     * <br>
     * Note that the keys may be revoked or expired - it is up to the client to handle such conditions.
     *
     * @param email The email
     * @param userId the Sender userId
     * @param cid the Sender context Id
     * @param timeout Procedure should timeout after specified milliseconds
     * @return A list of public keys, or an empty list if no keys were found for the given email
     * @throws OXException
     */
    Collection<WKSResult> findAll(String email, int userId, int cid, int timeout) throws OXException;

    /**
     * Tries to find all (including expired and revoked) public-key rings for the given email address by performing a WKS lookup on a specified, alternative, host/port.
     * <br>
     * <br>
     * The query is performed on the host obtained from the domain part of the given email address.
     * <br>
     * <br>
     * Note that the keys may be revoked or expired - it is up to the client to handle such conditions.
     *
     * @param email The email
     * @param userId the Sender userId
     * @param cid the Sender context Id
     * @param timeout Procedure should timeout after specified milliseconds
     * @param host The alternative host to perform the key lookup on
     * @param port The alternative port to use
     * @return A list of public keys, or an empty list if no keys were found for the given email
     * @throws OXException
     */
    Collection<WKSResult> findAll(String email, int userId, int cid, int timeout, String host, int port) throws OXException;

    /**
     * Tries to find the most usable (non revoked, non expired) public-key ring for the given email address by performing a WKS lookup.
     *
     * @param email The email
     * @param userId the Sender userId
     * @param cid the Sender context Id
     * @param timeout Procedure should timeout after specified milliseconds
     * @return The first suitable key found, or an empty Optional if no suitable key was found
     * @throws OXException
     */
    Optional<WKSResult> find(String email, int userId, int cid, int timeout) throws OXException;

    /**
     * Tries to find the most usable (non revoked, non expired) public-key ring for the given email address by performing a WKS lookup on the specified host/port.
     *
     * @param email The email
     * @param timeout Procedure should timeout after specified milliseconds
     * @param userId the Sender userId
     * @param cid the Sender context Id
     * @param host The alternative host to perform the key lookup on
     * @param port The alternative port to use
     * @return The first suitable key found, or an empty Optional if no suitable key was found
     * @throws OXException
     */
    Optional<WKSResult> find(String email, int userId, int cid, int timeout, String host, int port) throws OXException;
}
