/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services.impl;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.List;
import org.bouncycastle.openpgp.PGPSecretKey;
import com.openexchange.exception.OXException;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.pgpcore.services.GuardKeyRingRetrievalStrategy;
import com.openexchange.guard.pgpcore.services.PGPSigningService;
import com.openexchange.guard.pgpcore.services.exceptions.PGPCoreServicesExceptionCodes;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.core.PGPKeyRetrievalStrategy;
import com.openexchange.pgp.core.PGPSignatureCreator;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;
import com.openexchange.pgp.core.PGPSignatureVerifier;
import com.openexchange.pgp.keys.tools.PGPKeysUtil;

/**
 * {@link PGPSigningServiceImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.4.2
 */
public class PGPSigningServiceImpl implements PGPSigningService {

    private final PGPKeyRetrievalStrategy keyRetrievalStrategy;
    private final GuardKeyRingRetrievalStrategy recipientKeyRingRetrievalStrategy;

    /**
     * Initializes a new {@link PGPSigningServiceImpl}.
     *
     * @param guardKeyService
     */
    public PGPSigningServiceImpl(GuardKeyRingRetrievalStrategy recipientKeyRingRetrievalStrategy, PGPKeyRetrievalStrategy keyRetrievalStrategy) {
        this.recipientKeyRingRetrievalStrategy = recipientKeyRingRetrievalStrategy;
        this.keyRetrievalStrategy = keyRetrievalStrategy;
    }

    /**
     * Fetches the signing key for the given identity
     *
     * @param identity The identity to get the signing key for
     * @return The signing key for the given identity
     * @throws OXException If the key was not found
     */
    GuardKeys getSignerGuardKey(String identity) throws OXException {
        GuardKeys guardKeys  = recipientKeyRingRetrievalStrategy.getKeyRingForRecipient(identity);
        if (guardKeys != null) {
            return guardKeys;
        }
        else {
            throw PGPCoreServicesExceptionCodes.SIGNING_KEY_NOT_FOUND.create(identity);
        }
    }

    /**
     * Extracts the PGP signing key from the given Guard key
     *
     * @param guardKey The key to get the singing key for
     * @return The signing key for the given Guard key
     * @throws OXException If no signing key was found
     */
    PGPSecretKey getSignerKey(GuardKeys guardKey) throws OXException {
        PGPSecretKey signingKey = PGPKeysUtil.getSigningKey(guardKey.getPGPSecretKeyRing());
        if (signingKey != null) {
            return signingKey;
        }
        else {
            throw PGPCoreServicesExceptionCodes.SIGNING_KEY_NOT_FOUND.create(guardKey.getKeyid());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgpcore.services.PGPSigningService#sign(java.io.InputStream, java.io.OutputStream, boolean, org.bouncycastle.openpgp.PGPSecretKey, char[])
     */
    @Override
    public void sign(InputStream input, OutputStream output, boolean armored, UserIdentity signer) throws OXException {
        GuardKeys signerGuardKey = getSignerGuardKey(signer.getIdentity());
        sign(input, output, armored, signerGuardKey, signer.getPassword());
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.pgpcore.services.PGPSigningService#sign(java.io.InputStream, java.io.OutputStream, boolean, com.openexchange.guard.keymanagement.commons.GuardKeys, char[])
     */
    @Override
    public void sign(InputStream input, OutputStream output, boolean armored, GuardKeys signerKey, char[] password) throws OXException {
        try {
            PGPSecretKey signingKey = getSignerKey(signerKey);
            char[] hashedPassword = CipherUtil.getSHA(new String(password), signerKey.getSalt()).toCharArray();
            new PGPSignatureCreator().createSignature(input, output, armored, signingKey, hashedPassword);
        } catch (OXException e) {
            throw e;
        } catch (Exception e) {
            throw PGPCoreServicesExceptionCodes.SIGNING_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.pgpcore.services.PGPSigningService#verify(java.io.InputStream, java.io.InputStream)
     */
    @Override
    public List<PGPSignatureVerificationResult> verify(InputStream signedData, InputStream signatureData) throws OXException {
        try {
            return new PGPSignatureVerifier(keyRetrievalStrategy).verifySignatures(signedData, signatureData);
        } catch (Exception e) {
            throw PGPCoreServicesExceptionCodes.SIGNATURE_VERIFICATION_ERROR.create(e, e.getMessage());
        }
    }
}
