/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.pgpcore.services.impl;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Objects;
import org.bouncycastle.openpgp.PGPPublicKey;
import org.bouncycastle.openpgp.PGPPublicKeyRing;
import org.bouncycastle.openpgp.PGPSignature;
import com.openexchange.exception.OXException;
import com.openexchange.guard.autocrypt.database.AutoCryptStorageService;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.storage.OGPGPKeysStorage;
import com.openexchange.guard.pgpcore.services.PGPKeySigningService;
import com.openexchange.guard.pgpcore.services.exceptions.PGPCoreServicesExceptionCodes;
import com.openexchange.guard.user.UserIdentity;
import com.openexchange.pgp.core.PGPKeyRetrievalStrategy;
import com.openexchange.pgp.core.PGPKeySignatureVerifier;
import com.openexchange.pgp.core.PGPSignatureVerificationResult;

/**
 * {@link PGPKeySigningServiceImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v7.10.0
 */
public class PGPKeySigningServiceImpl implements PGPKeySigningService {

    private final GuardKeyService guardKeyService;
    private final OGPGPKeysStorage ogPGPKeysStorage;
    private final AutoCryptStorageService autocryptStorageService;

    /**
     * Initializes a new {@link PGPKeySigningServiceImpl}.
     *
     * @param guardKeyService
     * @param ogPGPKeysStorage
     * @param autocryptStorageService
     */
    public PGPKeySigningServiceImpl(GuardKeyService guardKeyService, OGPGPKeysStorage ogPGPKeysStorage,
        AutoCryptStorageService autocryptStorageService) {
        this.guardKeyService = guardKeyService;
        this.ogPGPKeysStorage = ogPGPKeysStorage;
        this.autocryptStorageService = autocryptStorageService;
    }

    private PGPKeyRetrievalStrategy getPGPKeyRetrievalStrategyFor(UserIdentity verifier) {
        return new DefaultPGPKeyRetrievalStrategy(guardKeyService,
            ogPGPKeysStorage,
            autocryptStorageService,
            verifier.getOXUser().getId(),
            verifier.getOXUser().getContextId());
    }

    /**
     * Internal method to verify a signature
     *
     * @param signature The signature to verify.
     * @param keyRetrievalStrategy The strategy to retrieval the public key for signature verification
     * @param publicKey The public key containing the given signature
     * @return The verification result
     * @throws Exception
     */
    private PGPSignatureVerificationResult verifySignature(PGPSignature signature, PGPKeyRetrievalStrategy keyRetrievalStrategy, PGPPublicKey publicKey) throws Exception {
        return new PGPKeySignatureVerifier(keyRetrievalStrategy).verifySignatures(signature, publicKey);
    }

    /**
     * Internal method to verify all signatures contained in the given public key.
     *
     * @param keyRetrievalStrategy The strategy to retrieval the public key for signature verification
     * @param publicKey The public key containing the signatures to verify.
     * @return The verification result
     * @throws Exception
     */
    private Collection<PGPSignatureVerificationResult> verifySignaturesFor(PGPKeyRetrievalStrategy keyRetrievalStrategy, PGPPublicKey publicKey) throws Exception {
        ArrayList<PGPSignatureVerificationResult> ret = new ArrayList<>();
        @SuppressWarnings("unchecked") Iterator<PGPSignature> signatures = publicKey.getSignatures();
        while (signatures.hasNext()) {
            ret.add(verifySignature(signatures.next(), keyRetrievalStrategy, publicKey));
        }
        return ret;
    }

    /**
     * Internal method to verify all signatures within the given public keys
     *
     * @param keyRetrievalStrategy The strategy to retrieval the public key for signature verification
     * @param publicKeys The public keys containing the signatures to verify.
     * @return The verification results
     * @throws Exception
     */
    private Collection<PGPSignatureVerificationResult> verifySignaturesFor(PGPKeyRetrievalStrategy keyRetrievalStrategy, Iterator<PGPPublicKey> publicKeys) throws Exception {
        ArrayList<PGPSignatureVerificationResult> ret = new ArrayList<>();
        while (publicKeys.hasNext()) {
            ret.addAll(verifySignaturesFor(keyRetrievalStrategy, publicKeys.next()));
        }
        return ret;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.pgpcore.services.PGPKeySigningService#verifyKey(com.openexchange.guard.user.UserIdentity, com.openexchange.guard.keymanagement.commons.GuardKeys)
     */
    @Override
    public Collection<PGPSignatureVerificationResult> verifyKey(UserIdentity verifier, GuardKeys keys) throws OXException {
        keys = Objects.requireNonNull(keys, "keys must not be null");
        return verifyKey(verifier, keys.getPGPPublicKeyRing());
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.pgpcore.services.PGPKeySigningService#verifyKey(com.openexchange.guard.user.UserIdentity, org.bouncycastle.openpgp.PGPPublicKeyRing)
     */
    @Override
    public Collection<PGPSignatureVerificationResult> verifyKey(UserIdentity verifier, PGPPublicKeyRing keyRing) throws OXException {
        keyRing = Objects.requireNonNull(keyRing, "keyRing must not be null");
        try {
            return verifySignaturesFor(getPGPKeyRetrievalStrategyFor(verifier), keyRing.getPublicKeys());
        } catch (Exception e) {
            throw PGPCoreServicesExceptionCodes.SIGNATURE_VERIFICATION_ERROR.create(e);
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.pgpcore.services.PGPKeySigningService#verifyKey(com.openexchange.guard.user.UserIdentity, org.bouncycastle.openpgp.PGPPublicKey)
     */
    @Override
    public Collection<PGPSignatureVerificationResult> verifyKey(UserIdentity verifier, PGPPublicKey key) throws OXException {
        key = Objects.requireNonNull(key, "key must not be null");
        try {
            return verifySignaturesFor(getPGPKeyRetrievalStrategyFor(verifier), key);
        } catch (Exception e) {
            throw PGPCoreServicesExceptionCodes.SIGNATURE_VERIFICATION_ERROR.create(e);
        }
    }
}
