/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.oxapi.guestUser;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;
import javax.servlet.http.HttpServletResponse;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpGet;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.exceptions.GuardCoreExceptionCodes;
import com.openexchange.guard.oxapi.AbstractOXCalls;
import com.openexchange.guard.oxapi.Api;
import com.openexchange.guard.oxapi.exceptions.OXApiExceptionCodes;
import com.openexchange.guard.oxapi.osgi.Services;
import com.openexchange.guard.oxapi.pooling.HttpConnectionPoolService;

/**
 * Looks up any OX userId associated with context and email
 * {@link OxLookup}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.5
 */
public class OxLookup extends AbstractOXCalls {

    private static Logger LOG = LoggerFactory.getLogger(OxLookup.class);
    private static final String SESSION_REST_PATH = "/preliminary/guard/guest/v1/getoxid";
    private static final String MIME_JSON = "application/json";
    private static final String RESULT_ID = "id";

    /**
     * Internal method to create a share link/guest
     *
     * @param url The full URL
     * @return The result object
     * @throws OXException
     * @throws UnsupportedEncodingException
     * @throws JSONException
     */
    private JSONObject sendRequest(String url) throws OXException, UnsupportedEncodingException, JSONException {
        HttpConnectionPoolService poolService = Services.getService(HttpConnectionPoolService.class);
        HttpClient httpClient = poolService.getClient();
        HttpGet get = new HttpGet(url);
        //get.addHeader("Content-Type", MIME_JSON);
        get.addHeader("accept", MIME_JSON);
        get.setHeader("User-Agent", Api.USER_AGENT);
        get.addHeader(getAuthenticationHeader());
        HttpResponse response = null;
        try {
            response = httpClient.execute(get);
            try (BufferedReader reader = new BufferedReader(new InputStreamReader(response.getEntity().getContent(), "UTF-8"))) {
                int statusCode = response.getStatusLine().getStatusCode();
                if (statusCode == HttpServletResponse.SC_OK) {
                    return new JSONObject(reader);
                } else {
                    LOG.error(response.toString());
                    throw OXApiExceptionCodes.WRONG_ERROR_CODE.create(statusCode);
                }
            }
        } catch (IOException e) {
            throw GuardCoreExceptionCodes.IO_ERROR.create(e, e.getMessage());
        } finally {
            get.releaseConnection();
        }
    }

    /**
     * Creates the URL to send
     * createURL
     *
     * @param email Email address of the guest
     * @param cid ContextID of the guest
     * @return
     * @throws OXException
     */
    private String createURL(String email, int cid) throws OXException {
        return getUrl() + "/" + cid + "/" + email;
    }

    /**
     * Creates the full URL to call
     *
     * @return The URL to call
     * @throws OXException
     */
    private String getUrl() throws OXException {
        return getMainURI() + SESSION_REST_PATH;
    }

    public int getId(String email, int cid) {
        int id = 0;
        try {
            JSONObject json = sendRequest(createURL(email, cid));
            if (json.has(RESULT_ID)) {
                id = json.getInt(RESULT_ID);
            }
        } catch (UnsupportedEncodingException | OXException | JSONException e) {
            LOG.error("Error gettingg guest ID: ", e);
        }
        return id;
    }

}
