/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services;

import java.security.Key;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.MasterKey;

/**
 * {@link MasterKeyService}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.6
 */
public interface MasterKeyService {

    /**
     * Gets the master key for a specific user index
     * getMasterKey
     *
     * @param userId id of the user
     * @param cid cid of the user
     * @return
     * @throws OXException
     */
    default MasterKey getMasterKey(int userId, int cid) throws OXException {
        return getMasterKey(userId, cid, false);
    }

    /**
     * Gets the master key for a specific user index
     * getMasterKey
     *
     * @param index of the key
     * @param recovery True if keys to decrypt password recovery required
     * @return
     * @throws OXException
     */
    public MasterKey getMasterKey(int index, boolean recovery) throws OXException;

    /**
     * Gets the master key for a specific user index
     * getMasterKey
     *
     * @param userId id of the user
     * @param cid cid of the user
     * @param recovery True if keys to decrypt password recovery required
     * @return
     * @throws OXException
     */
    public MasterKey getMasterKey(int userId, int cid, boolean recovery) throws OXException;

    /**
     * Gets the index for a specific user
     * getIndexForUser
     *
     * @param userId
     * @param cid
     * @return
     * @throws OXException
     */
    public int getIndexForUser(int userId, int cid) throws OXException;

    /**
     * getDecryptedClientKey
     *
     * @param key
     * @return
     * @throws OXException
     */
    Key getDecryptedClientKey(MasterKey key) throws OXException;

    /**
     * Encrypts data with the AES rc pass
     * getRcEncryted
     *
     * @param data
     * @param salt
     * @param index
     * @return
     * @throws OXException
     */
    String getRcEncryted(String data, String salt, int index) throws OXException;

    /**
     * Decrypts data with the AES rc pass
     * getRcDecrypted
     *
     * @param data
     * @param salt
     * @param index
     * @return
     * @throws OXException
     */
    String getRcDecrypted(String data, String salt, int index) throws OXException;

    /**
     * Creates a new masterKey pair and stores it in the implementation specific key storage.
     *
     * @param index The index to create the key for
     * @return The {@link MasterKey} The created and stored key
     * @throws OXException due an error while creating or storing the key
     */
    MasterKey createKey() throws OXException;

    /**
     * Creates a new masterKey pair for the specified index
     * and stores it in the implementation specific key storage.
     *
     * @param index The index to create the key for
     * @return The {@link MasterKey} The created and stored key
     * @throws OXException due an error while creating or storing the key
     */
    MasterKey createKey(int index) throws OXException;
}
