/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl.masterKey;

import java.security.Key;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.keymanagement.commons.MasterKey;
import com.openexchange.guard.keymanagement.services.MasterKeyService;

import static com.openexchange.java.Autoboxing.I;

/**
 * Retrieve master keys from cache if available, otherwise use delegate.
 * Master keys with the recovery master keys are NOT cached.
 * {@link CachingMasterKeyService}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.6
 */
public class CachingMasterKeyService extends AbstractMasterKeyService {

    private final Map<Integer, MasterKey> map;  // Simple cache of index to master key.  Will be small sized and fairly static
    private final MasterKeyService delegate;

    public CachingMasterKeyService(MasterKeyService delegate, GuardCipherService cipher) {
        super(cipher);
        this.map = new ConcurrentHashMap<Integer, MasterKey>();
        this.delegate = delegate;
    }

    @Override
    public MasterKey getMasterKey(int userId, int cid, boolean recovery) throws OXException {
        final int index = getIndexForUser(userId, cid);
        return getMasterKey(index, recovery);
    }

    @Override
    public int getIndexForUser(int userId, int cid) throws OXException {
        return delegate.getIndexForUser(userId, cid);
    }

    @Override
    public MasterKey getMasterKey(int index, boolean recovery) throws OXException {
        if (recovery) {  // We don't cache the password recovery key
            return delegate.getMasterKey(index, recovery);
        }
        if (map.containsKey(I(index))) {
            return map.get(I(index));
        }
        MasterKey key = delegate.getMasterKey(index, false);
        if (key != null) {
            map.put(I(index), key);
        }
        return key;
    }

    @Override
    public Key getDecryptedClientKey(MasterKey key) throws OXException {
        return delegate.getDecryptedClientKey(key);
    }

    @Override
    public String getRcEncryted(String data, String salt, int index) throws OXException {
        return encrypt(getMasterKey(index, false), data, salt);
    }

    @Override
    public String getRcDecrypted(String data, String salt, int index) throws OXException {
        return decrypt(getMasterKey(index, false), data, salt);
    }

    @Override
    public MasterKey createKey(int index) throws OXException {
        return delegate.createKey(index);
    }

    @Override
    public MasterKey createKey() throws OXException {
        return delegate.createKey();
    }
}
