/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl.lookup;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.lookup.RecipKeyLookupStrategy;

/**
 * {@link DefaultRecipKeyLookupStrategy} defines a concrete and the default {@link RecipKeyLookupStrategy} used by OX Guard
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class DefaultRecipKeyLookupStrategy implements RecipKeyLookupStrategy {

    private final int senderUserId;
    private final int senderContextId;
    private final String hkpClientToken;
    private static final Logger logger = LoggerFactory.getLogger(DefaultRecipKeyLookupStrategy.class);

    /**
     * Initializes a new {@link DefaultRecipKeyLookupStrategy}.
     *
     * @param senderUserId
     * @param senderContextId
     */
    public DefaultRecipKeyLookupStrategy(int senderUserId, int senderContextId, String hkpClientToken) {
        this.hkpClientToken = hkpClientToken;
        this.senderUserId = senderUserId;
        this.senderContextId = senderContextId;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keymanagement.services.lookup.RecipKeyLookupStrategy#lookup(java.lang.String)
     */
    @Override
    public RecipKey lookup(String email, int timeout) throws OXException {

        RecipKey recipKey = new CompositeRecipKeyLookupStrategy(new RecipKeyLookupStrategy[] {
            new UserUploadedRecipKeyLookupStrategy(senderUserId, senderContextId),
            new GuardRecipKeyLookupStrategy(senderUserId, senderContextId),
            new MailResolverRecipKeyLookupStrategy(),
            new AutoCryptLookupStrategy(senderUserId, senderContextId),
            new WKSRecipKeyLookupStrategy(senderUserId, senderContextId),
            new HKPRecipKeyLookupStrategy(hkpClientToken, senderUserId, senderContextId),
            new GuestRecipKeyLookupStrategy(senderUserId, senderContextId),
        }).lookup(email, timeout);

        if(recipKey == null) {
            logger.info("Could not find a recipient for email {}", email);
        }
        else {
            if(recipKey.isNewKey()) {
                logger.info("Guard will create a new {} for email {}",
                    recipKey.isGuest() ? "guest key" : "key" ,
                    email);
            }
            else {
                logger.info("Found a recipient for email {}", email);
            }
        }
        return recipKey;
    }


}
