/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl;

import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.common.java.Strings;
import com.openexchange.guard.common.util.CipherUtil;
import com.openexchange.guard.configuration.GuardConfigurationService;
import com.openexchange.guard.configuration.GuardProperty;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.MasterKey;
import com.openexchange.guard.keymanagement.services.KeyRecoveryService;
import com.openexchange.guard.keymanagement.services.MasterKeyService;
import com.openexchange.guard.keymanagement.services.impl.exceptions.GuardKeyTableExceptionCodes;
import com.openexchange.guard.keymanagement.services.impl.osgi.Services;
import com.openexchange.guard.user.GuardCapabilities;
import com.openexchange.guard.user.OXUserService;
import com.openexchange.server.ServiceLookup;

/**
 * {@link KeyRecoveryServiceImpl}
 *
 * @author <a href="mailto:benjamin.gruedelbach@open-xchange.com">Benjamin Gruedelbach</a>
 * @since v2.6.0
 */
public class KeyRecoveryServiceImpl implements KeyRecoveryService {

    ServiceLookup services;

    /**
     * Initializes a new {@link KeyRecoveryServiceImpl}.
     *
     * @param configurationService The configuration service
     * @param userService The user service
     */
    public KeyRecoveryServiceImpl(ServiceLookup services) {
        this.services = services;
    }

    private GuardConfigurationService getConfigService() throws OXException {
        return services.getServiceSafe(GuardConfigurationService.class);
    }


    private String createRecoveryInternal(GuardKeys keys, String password) throws OXException {
        MasterKeyService masterKeyService = services.getServiceSafe(MasterKeyService.class);
        MasterKey mkey = masterKeyService.getMasterKey(keys.getMasterKeyIndex(), true);
        String recovery = "";
        GuardCipherService aesCipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        String encrPass = aesCipherService.encrypt(password, mkey.getRC() + (keys.getPublicKey() == null ? "" : Integer.toString(keys.getEncodedPublic().hashCode())), keys.getSalt());
        GuardCipherService rsaCipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.RSA);
        recovery = rsaCipherService.encrypt(encrPass, mkey.getMasterPublic());


        if (Strings.isEmpty(recovery) && keys.getUserid() > 0) {  // make sure not for system email
            throw GuardKeyTableExceptionCodes.UNABLE_TO_CREATE_RECOVERY.create(keys.getUserid(), keys.getContextid(), keys.getEmail());
        }
        return recovery;
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.KeyRecoveryService#checkCreateRecovery(int, int)
     */
    @Override
    public boolean checkCreateRecovery(int contextId, int userId) throws OXException {
        if (getConfigService().getBooleanProperty(GuardProperty.noRecovery, userId, contextId)) {
            // If global no Recovery then return false.
            return false;
        }
        if(contextId < 0) {
            return true;
        }

        // get individual settings
        OXUserService userService = services.getServiceSafe(OXUserService.class);
        GuardCapabilities userCapabilities = userService.getGuardCapabilieties(contextId, userId);
        return !userCapabilities.isNorecovery();
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.keymanagement.services.KeyRecoveryService#createRecovery(com.openexchange.guard.keymanagement.commons.GuardKeys, java.lang.String)
     */
    @Override
    public String createRecovery(GuardKeys keys, String password) throws OXException {
        if (Strings.isEmpty(password)) {
            return "-1";
        }
        password = CipherUtil.getSHA(password, keys.getSalt());
        return createRecoveryInternal(keys, password);
    }
}
