/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.keymanagement.services.impl;

import java.util.Objects;
import com.openexchange.exception.OXException;
import com.openexchange.guard.cipher.GuardCipherAlgorithm;
import com.openexchange.guard.cipher.GuardCipherFactoryService;
import com.openexchange.guard.cipher.GuardCipherService;
import com.openexchange.guard.keymanagement.commons.GuardKeys;
import com.openexchange.guard.keymanagement.commons.RecipKey;
import com.openexchange.guard.keymanagement.services.AccountCreationService;
import com.openexchange.guard.keymanagement.services.GuardKeyService;
import com.openexchange.guard.keymanagement.services.KeyCreationService;
import com.openexchange.guard.keymanagement.services.impl.exceptions.AccountCreationExceptionCodes;
import com.openexchange.guard.keymanagement.services.impl.osgi.Services;

public class AccountCreationServiceImpl implements AccountCreationService {

    private RecipKey createUser(RecipKey recipKey) throws OXException {

        GuardCipherService cipherService = Services.getService(GuardCipherFactoryService.class).getCipherService(GuardCipherAlgorithm.AES_CBC);
        KeyCreationService keyCreationService = Services.getService(KeyCreationService.class);
        GuardKeyService keyService = Services.getService(GuardKeyService.class);
        if (!recipKey.isNewKey()) {
            throw AccountCreationExceptionCodes.ACCOUNT_ALREADY_EXISTS.create();
        }

        // When creating a new key for a recipient, we will always create a recovery in case of lost email, etc
        final boolean createRecovery = true;
        final boolean markNewKeyAsCurrent = true;
        final String newRandomPassword = cipherService.generateRandomPassword(recipKey.getSenderUserId(), recipKey.getSenderCid());
        //Create
        GuardKeys keys = keyCreationService.create(recipKey.getUserid(),
            recipKey.getCid(),
            recipKey.getName(),
            recipKey.getEmail(),
            newRandomPassword,
            recipKey.getLocale(),
            markNewKeyAsCurrent,
            createRecovery,
            recipKey.getSenderUserId(),
            recipKey.getSenderCid());
        //Store
        final boolean isUserCreatedKey = false;
        keys = keyService.storeKeys(keys, isUserCreatedKey);
        keyService.setCurrentKey(keys);

        //Setting the created keys and return them
        recipKey.setPGPPublicKeyRing(keys.getPGPPublicKeyRing());
        recipKey.setPubkey(keys.getPublicKey());
        if (recipKey.isGuest()) {
            recipKey.setUserid(keys == null ? -1 : keys.getUserid());
            recipKey.setNewGuestPass(newRandomPassword);
        }
        recipKey.setPgp(true);
        //Mark the key as new created key
        recipKey.setNewCreated();
        return recipKey;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.guard.keymanagement.services.AccountCreationService#createUserFor(com.openexchange.guard.keymanagement.commons.RecipKey)
     */
    @Override
    public RecipKey createUserFor(RecipKey recipKey) throws OXException {
        recipKey = Objects.requireNonNull(recipKey, "recipKey must not be null");
        return createUser(recipKey);
    }
}
