/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.database.internal;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import com.openexchange.exception.OXException;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.exception.GuardDatabaseExceptionCodes;
import com.openexchange.guard.database.utils.DBUtils;

/**
 * {@link GuardShardingIDManagement}  Class for keeping guest shard Id autoincrements up to date and
 * correct
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.10.3
 */
public class GuardShardingIDManagement {

    private static final org.slf4j.Logger LOG = org.slf4j.LoggerFactory.getLogger(GuardShardingIDManagement.class);

    GuardDatabaseService guardDatabaseService;
    public GuardShardingIDManagement (GuardDatabaseService guardDatabaseService) {
        this.guardDatabaseService = guardDatabaseService;
    }

    private static final String SELECT_HIGHEST_GUEST_USERID_STMT = "SELECT MAX(ID) FROM og_email WHERE cid < 0";
    private static final String ALTER_AUTOINCREMENT_STMT = "ALTER TABLE `og_KeyTable` AUTO_INCREMENT=?";
    private static final String GET_AUTOINCREMENT_STMT = "SELECT `AUTO_INCREMENT` FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = ? AND TABLE_NAME = 'og_KeyTable';";
    private static final String LOCK_TABLE = "LOCK TABLES `og_KeyTable` WRITE;";
    private static final String UNLOCK = "UNLOCK TABLES";

    /**
     * Lock to og_KeyTable while autoincrement is being set.
     *
     * @param con  Database connection
     * @throws OXException
     */
    private void lockTables (Connection con) throws OXException {
        PreparedStatement stmt = null;
        try {
            stmt = con.prepareStatement(LOCK_TABLE);
            stmt.execute();
        } catch (SQLException e) {
                throw GuardDatabaseExceptionCodes.DB_ERROR.create(e,e.getMessage());
            } finally {
                DBUtils.closeSQLStuff(stmt);
            }
    }

    /**
     * Unlock tables
     *
     * @param con  Database connection
     * @throws OXException
     */
    private void unLockTables (Connection con) throws OXException {
        PreparedStatement stmt = null;
        try {
            stmt = con.prepareStatement(UNLOCK);
            stmt.execute();
        } catch (SQLException e) {
                throw GuardDatabaseExceptionCodes.DB_ERROR.create(e,e.getMessage());
            } finally {
                DBUtils.closeSQLStuff(stmt);
            }
    }

    /**
     * Update the shard's autoincrement value to the highest Guest Id
     *
     * @param shardId
     * @throws OXException
     */
    public void updateAutoIncrement (int shardId) throws OXException {
        Connection connection = guardDatabaseService.getWritableForShard(shardId);
        try {
            // First, lock the og_KeyTable.  The getMaxGuest id may take a moment, and we don't
            // want other writes
            lockTables(connection);
            // Update the autoincrement with max Guest + 1
            updateAutoIncrement(connection, shardId, getMaxGuestId() + 1);
        } finally {
            try {
                // Unlock
                unLockTables(connection);
            } catch (OXException e) {
                LOG.error("Error unlocking Guest shard og_KeyTable ", e);
            }
            guardDatabaseService.backReadOnlyForGuard(connection);
        }

    }

    /**
     * Gets the highest Guest Id from the og_email table
     *
     * @return Highest guest ID
     * @throws OXException
     */
    private int getMaxGuestId () throws OXException {
        Connection connection = guardDatabaseService.getReadOnlyForGuard();

        PreparedStatement stmt = null;
        ResultSet resultSet = null;
        try {
            stmt = connection.prepareStatement(SELECT_HIGHEST_GUEST_USERID_STMT);
            resultSet = stmt.executeQuery();

            int max = 1;
            if (resultSet.next()) {
                max = resultSet.getInt(1);
            }
            return max;
        } catch (SQLException e) {
            throw GuardDatabaseExceptionCodes.DB_ERROR.create(e, e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(resultSet, stmt);
            guardDatabaseService.backReadOnlyForGuard(connection);
        }
    }

    /**
     * Update the autoincrement value of the og_KeyTable for a shard
     *
     * @param shardId  Id of the shard
     * @param value  New value
     * @throws OXException
     */
    private void updateAutoIncrement(Connection connection, int shardId, int value) throws OXException {
        PreparedStatement stmt = null;

        try {
            stmt = connection.prepareStatement(ALTER_AUTOINCREMENT_STMT);
            stmt.setInt(1, value);

            stmt.executeUpdate();
        } catch (SQLException e) {
            throw GuardDatabaseExceptionCodes.DB_ERROR.create(e,e.getMessage());
        } finally {
            DBUtils.closeSQLStuff(stmt);
        }
    }

}
