/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.autocrypt.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import org.bouncycastle.openpgp.PGPEncryptedData;
import org.bouncycastle.openpgp.PGPException;
import org.bouncycastle.openpgp.PGPObjectFactory;
import org.bouncycastle.openpgp.PGPUtil;
import org.bouncycastle.openpgp.operator.bc.BcKeyFingerprintCalculator;
import com.openexchange.pgp.core.PGPSymmetricDecrypter;
import com.openexchange.pgp.core.PGPSymmetricEncrypter;

/**
 * {@link AutoCryptCrypto}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.2
 */
public class AutoCryptCrypto {

    private static final boolean ASCII_ARMORED = true;

    /**
     * Performs the decryption of symmetric encrypted PGP data
     *
     * @param in The symmetric encrypted PGP data
     * @param out The stream to write the decrypted data to
     * @param password The password required for decryption
     * @throws IOException
     * @throws PGPException
     */
    public static void decryptSymmetric(InputStream in, OutputStream out, char[] password) throws IOException, PGPException {
        new PGPSymmetricDecrypter().decrypt(in, out, password);
    }

    /**
     * Performs the decryption of one single, symmteric encrypted PGP Object
     *
     * @param in The symmetric encrypted PGP object data
     * @param password The password
     * @return The decrypted PGP object
     * @throws IOException
     * @throws PGPException
     */
    @SuppressWarnings("unchecked")
    public static <T> T decryptSymmetricPGPObject (InputStream in, char[] password) throws IOException, PGPException {
        ByteArrayOutputStream decrypted = new ByteArrayOutputStream();
        decryptSymmetric(in, decrypted, password);
        ByteArrayInputStream pgpDataInputStream = new ByteArrayInputStream(decrypted.toByteArray());
        PGPObjectFactory factory = new PGPObjectFactory(PGPUtil.getDecoderStream(pgpDataInputStream), new BcKeyFingerprintCalculator());
        try {
            return (T)factory.nextObject();
        }
        catch(ClassCastException c) {
           throw new PGPException("Unable to cast PGP object");
        }
    }

    /**
     * Performs symmetric PGP encryption of data using AES-256
     *
     * @param data The data to encrypt
     * @param out the stream to write the encrypted data to
     * @param key The symmetric key
     * @throws IOException
     * @throws PGPException
     */
    public static void encryptSymmetric(InputStream data, OutputStream out, char[] key) throws IOException, PGPException {
        encryptSymmetric(data, out, PGPEncryptedData.AES_256, key);
    }

    /**
     * Performs symmetric PGP encryption of data
     *
     * @param data The data to encrypt
     * @param out the stream to write the encrypted data to
     * @param algorithm the ID of the symmetric algorithm to use (See RFC-4880 - 9.2 Symmetric key algorithm)
     * @param key The symmetric key
     * @throws IOException
     * @throws PGPException
     */
    public static void encryptSymmetric(InputStream data, OutputStream out, int algorithm, char[] key) throws IOException, PGPException {
        new PGPSymmetricEncrypter(algorithm).encrypt(data, out, ASCII_ARMORED, key);

    }
}
