/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.activity.impl;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.guard.activity.ActivityTrackingService;
import com.openexchange.guard.database.GuardDatabaseService;
import com.openexchange.guard.database.utils.DBUtils;
import com.openexchange.guard.user.OXGuardUser;

/**
 * {@link ActivityTrackingServiceImpl}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v2.10.0
 */
public class ActivityTrackingServiceImpl implements ActivityTrackingService {

    private final GuardDatabaseService guardDatabaseService;
    final Logger logger = LoggerFactory.getLogger(ActivityTrackingServiceImpl.class);

    public ActivityTrackingServiceImpl(GuardDatabaseService guardDatabaseService) {
        this.guardDatabaseService = guardDatabaseService;
    }
    /* (non-Javadoc)
     * @see com.openexchange.guard.activity.ActivityTrackingService#updateActivity(int, int)
     */
    @Override
    public void updateActivity(int userId, int contextId) {
        Connection connection = null;
        PreparedStatement stmt = null;
        try {
            connection = guardDatabaseService.getWritableForGuard();
            stmt = connection.prepareStatement(ActivityTrackingSql.UPDATE_ACTIVITY);
            stmt.setInt(1, userId);
            stmt.setInt(2, contextId);
            stmt.execute();
        } catch (OXException | SQLException ex) {
            logger.error("Problem saving activity", ex);
        } finally {
            DBUtils.closeSQLStuff(stmt);
            if (connection != null) {
                guardDatabaseService.backWritableForGuard(connection);
            }
        }
    }

    /* (non-Javadoc)
     * @see com.openexchange.guard.activity.ActivityTrackingService#removeActivityRecord(int, int)
     */
    @Override
    public void removeActivityRecord(int userId, int contextId) {
        Connection connection = null;
        PreparedStatement stmt = null;
        try {
            connection = guardDatabaseService.getWritableForGuard();
            stmt = connection.prepareStatement(ActivityTrackingSql.REMOVE_ACTIVITY);
            stmt.setInt(1, userId);
            stmt.setInt(2, contextId);
            stmt.execute();
        } catch (OXException | SQLException ex) {
            logger.error("Problem saving activity", ex);
        } finally {
            DBUtils.closeSQLStuff(stmt);
            if (connection != null) {
                guardDatabaseService.backWritableForGuard(connection);
            }
        }

    }
    /* (non-Javadoc)
     * @see com.openexchange.guard.activity.ActivityTrackingService#getExpiredGuests()
     */
    @Override
    public List<OXGuardUser> getExpiredGuests(int days) {
        Connection connection = null;
        PreparedStatement stmt = null;
        ResultSet rs = null;
        ArrayList<OXGuardUser> users = new ArrayList<OXGuardUser>();
        try {
            connection = guardDatabaseService.getReadOnlyForGuard();
            stmt = connection.prepareStatement(ActivityTrackingSql.GET_OLD_ITEMS);
            stmt.setInt(1, days);
            rs = stmt.executeQuery();
            while (rs.next()) {
                int userId = rs.getInt(1);
                int contextId = rs.getInt(2);
                users.add(new OXGuardUser(userId, contextId));
            }
            return users;
        } catch (OXException | SQLException ex) {
            logger.error("Problem saving activity", ex);
            return users;
        } finally {
            DBUtils.closeSQLStuff(rs);
            DBUtils.closeSQLStuff(stmt);
            if (connection != null) {
                guardDatabaseService.backReadOnlyForGuard(connection);
            }
        }
    }

}
