/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2017 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

/// <reference path="../../steps.d.ts" />

Feature('Send Encrypted To guest user');

const codecept = require('codeceptjs'),
    ox = codecept.config.get().helpers['OpenXchange'];

Before(async function (users, contexts) {
    await users.create();
    await contexts.create({ id: ox.newContext || 12, filestoreId: ox.filestoreId });
    const guestUser = await users.create(undefined, { id: ox.newContext || 12 });  // Two users
    await guestUser.hasAccessCombination('groupware_standard');
    await guestUser.doesntHaveCapability('guard');

});

After(async function (users, contexts, I) {
    await I.wipeUser(users).catch(e => console.log(e));;  // cleanup guest accounts
    await users.removeAll();
    await contexts.removeAll();
});

Scenario('Compose and email to guest user', async function (I, users) {

    const o1 = {
        user: users[0]
    };
    const o2 = {
        user: users[1]
    };

    await I.setupUser(o1, true);
    await I.setupUser(o2);

    const userdata1 = o1.user.userdata;
    const userdata2 = o2.user.userdata;

    await o2.user.doesntHaveCapability('guard');  // Non Guard user.  Guest only

    // Log in as User 0
    I.login('app=io.ox/mail', o1);

    I.waitForVisible('.io-ox-mail-window .window-body .classic-toolbar');

    I.verifyUserSetup(userdata1);  // Verify user has Guard setup

    var random = Math.floor((Math.random() * 100000));
    var subject = "Test Subject " + random;
    var data = "Test data " + random;

    // Compose email to Guest
    I.sendEmailGuest(userdata2, subject, data);

    I.logout();

    // Login to retrieve guest invite email
    I.login('app=io.ox/mail', o2);
    I.waitForVisible('.io-ox-mail-window .leftside');
    I.click('#io-ox-refresh-icon');
    I.waitForVisible('.io-ox-mail-window .leftside ul li.unread');
    I.click('.io-ox-mail-window .leftside ul li.unread');
    I.wait(1);
    I.waitForVisible('.mail-detail-frame');
    I.switchTo('.mail-detail-frame');
    I.see('Test introduction text');
    I.click('.bodyButton a');

    I.wait(2);
    // Opening Guest user tab
    await I.switchToNextTab();

    I.waitForVisible('.io-ox-mail-window', 30);

    I.changeTemporaryPassword(userdata2);

    //I.waitForVisible('.wizard-footer [data-action="done"]');
    //I.click('.wizard-footer [data-action="done"]');

    I.wait(1);

    // Confirm email decrypted
    I.waitForVisible('.mail-detail-frame');
    I.switchTo('.mail-detail-frame');
    I.see(data);
    I.switchTo();

    //Reply
    I.click('Reply');

    I.auth(userdata2);

    I.waitForVisible('.window-blocker.io-ox-busy');
    I.waitForInvisible('.window-blocker.io-ox-busy');
    // Check unable to add recipients
    
    I.dontSeeElementInDOM('.recipient-actions');

    // Change contents to ReplyTest to verify later
    await within({ frame: '.io-ox-mail-compose-window .editor iframe' }, async () => {
        I.see(data, '.mce-content-body');
        I.see('wrote:');
        I.click('.mce-content-body');
        I.fillField('.mce-content-body', 'ReplyTest');
      });

    I.waitForVisible('i.fa.fa-key.oxguard_token.key_tooltip.trusted_key');
    // Send
    I.click('Send');

    I.waitForVisible('.io-ox-mail-window .leftside');
    I.wait(3);
    I.waitForInvisible('.mail-progress');

    // Cleanup
    // Guest link logout will bring this tab back to the recipient tab logged in
    I.closeCurrentTab();
    I.wait(1);

    // Logout from Guest webmail
    I.logout();

    // Log in as User 0
    I.login('app=io.ox/mail', o1);

    // Verify sender can read the reply
    I.waitForVisible('.io-ox-mail-window .window-body .classic-toolbar');
    I.waitForVisible('.io-ox-mail-window .leftside ul li.unread');
    I.click('.io-ox-mail-window .leftside ul li.unread');
    I.waitForElement('.btn.oxguard_passbutton');
    I.waitForElement('input.form-control.password_prompt');
    I.insertCryptPassword('input.form-control.password_prompt', userdata1);
    I.click('.btn.oxguard_passbutton');

    I.verifyDecryptedMail(subject, 'ReplyTest');

    I.logout();

});
