/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * © 2017 OX Software GmbH, Germany. info@open-xchange.com
 *
 * @author Greg Hill <greg.hill@open-xchange.com>
 */

/// <reference path="../../steps.d.ts" />

Feature('PIN');

const codecept = require('codeceptjs'),
    ox = codecept.config.get().helpers['OpenXchange'];

Before(async function (users, contexts) {
    const ctx = ox.newContext || 12;
    await contexts.create({ id: ctx, filestoreId: ox.filestoreId }).then(function (context) {
        context.hasCapability('guard');
        context.hasCapability('guard-pin');
        context.hasCapability('guard-mail');
    });
    await users.create(undefined, { id: ctx })

    await contexts.create({ id: ctx + 1, filestoreId: ox.filestoreId  }).then(function (context) {
        context.doesntHaveCapability('guard');
        context.hasCapability('guard-pin');
    });
    await users.create(undefined, { id: ctx + 1 });  // Two users
    await users[1].hasConfig('com.openexchange.guard.pinEnabled', true);

});

After(async function (users, contexts, I) {
    await I.wipeUser(users).catch(e => console.log(e));;  // cleanup guest accounts
    await users.removeAll().catch(e => console.log(e));
    await contexts.removeAll().catch(e => console.log(e));
});

Scenario('Compose and email with PIN to guest user', async function (I, users) {

    const o1 = {
        user: users[0]
    };
    const o2 = {
        user: users[1]
    };

    await I.setupUser(o1, true);
    await I.setupUser(o2);

    const userdata1 = o1.user.userdata;
    const userdata2 = o2.user.userdata;

    await o2.user.doesntHaveCapability('guard');  // Non Guard user.  Guest only

    // Log in as User 0
    I.login('app=io.ox/mail', o1);

    I.waitForVisible('.io-ox-mail-window .window-body .classic-toolbar');

    I.verifyUserSetup(userdata1);  // Verify user has Guard setup

    var random = Math.floor((Math.random() * 100000));
    var subject = "Test Subject " + random;
    var data = "Test data " + random;

    // Compose email to Guest
    I.sendEmailGuest(userdata2, subject, data, true);

    I.waitForVisible('.pin');
    let pin = await I.grabTextFrom('.pin');

    I.click('Use PIN');
    I.waitForVisible('.io-ox-mail-window .leftside');
    I.wait(1);
    I.waitForInvisible('.mail-progress');

    I.selectFolder('Sent');
    I.waitForVisible('.io-ox-mail-window .leftside ul li.list-item', 20);
    I.click('.io-ox-mail-window .leftside ul li.list-item');

    I.waitForVisible('.io-ox-mail-window .mail-detail-pane .subject');
    I.waitForVisible('.thread-view .dropdown.more-dropdown a.dropdown-toggle');
    I.click('.thread-view .dropdown.more-dropdown a.dropdown-toggle');
    I.dontSee('PIN');
    I.click('.thread-view .dropdown.more-dropdown a.dropdown-toggle');
    I.waitForElement('input.form-control.password_prompt');
    I.insertCryptPassword('input.form-control.password_prompt', userdata1);
    I.click('.btn.oxguard_passbutton');
    I.verifyDecryptedMail(subject, data);


    I.click('.thread-view .dropdown.more-dropdown a.dropdown-toggle');
    I.see('Check assigned PIN');
    I.click('Check assigned PIN');

    I.see(pin, '.modal-body');
    I.click('OK');

    I.logout();

    // Login to retrieve guest invite email
    I.login('app=io.ox/mail', o2);
    I.waitForVisible('.io-ox-mail-window .leftside');
    I.waitForVisible('.io-ox-mail-window .leftside ul li.unread');
    I.click('.io-ox-mail-window .leftside ul li.unread');

    I.waitForVisible('.mail-detail-frame');
    I.switchTo('.mail-detail-frame');
    I.see('Test introduction text');
    I.wait(1);
    I.click('.bodyButton a');

    I.wait(1);
    // Opening Guest user tab
    await I.switchToNextTab();

    I.waitForVisible('.io-ox-mail-window .leftside', 30);

    I.changeTemporaryPassword(userdata2);
    I.waitForVisible('.modal-body');
    I.see('The pin is missing', '.modal-body');
    I.fillField('#pinInput', pin);
    I.click('.btn[data-action="okpass"]');
    I.wait(3);

    // Confirm email decrypted
    I.waitForVisible('.mail-detail-frame');
    I.switchTo('.mail-detail-frame');
    I.see(data);
    I.switchTo();

    // Cleanup
    // Guest link logout will bring this tab back to the recipient tab logged in
    I.logout();
    I.wait(1);
    I.closeCurrentTab();
    // Logout from Guest webmail
    I.logout();

});
