/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 * Author Greg Hill <greg.hill@open-xchange.com>
 */

define('oxguard/tour/main', [
    'io.ox/core/tk/wizard',
    'io.ox/core/tk/hotspot',
    'oxguard/oxguard_core',
    'oxguard/util',
    'io.ox/core/capabilities',
    'io.ox/backbone/mini-views/helplink',
    'oxguard/core/createKeys',
    'settings!oxguard',
    'gettext!oxguard/tour',
    'less!oxguard/tour/style'
], function (Tour, hotspot, core, util, capabilities, HelpView, keys, settings, gt) {

    'use strict';

    function writeMail() {
        ox.registry.call('mail-compose', 'open')
        .done(function (e) {
            e.app.view.securityModel.set('encrypt', true);
        });
    }

    function openFiles() {
        ox.launch('io.ox/files/main');
    }

    function openSettings() {
        ox.launch('io.ox/settings/main', { id: 'oxguard' });
    }
    function setEncryptDefault() {
        ox.launch('io.ox/settings/main', { id: 'oxguard' }).then(function () {
            if (!settings.get('defaultEncrypted')) {
                $('[name="defaultEncrypted"]').click();
            }
        });
    }

    Tour.registry.add({
        id: 'default/oxguard',
        priority: 1
    }, function () {
        if (!util.isGuardConfigured()) {
            Tour.registry.run('default/oxguard/createKeys');
            return;
        }

        var def = $.Deferred();
        var tour = new Tour(),
            composeApp;
        var driveText = '';
        var resetText = '';
        // Select text depending on if recovery is available
        if (window.oxguarddata.recoveryAvail) {
            resetText = gt.format(gt.pgettext('tour', 'You already set up your %s account. But you can always change your encryption password, or request a temporary password if you forget it.'), window.oxguarddata.productName);
        } else {
            resetText = gt.format(gt.pgettext('tour', 'You already set up your %s account.  You can change your encryption password here.'), window.oxguarddata.productName);
        }
        // minimze all open windows
        ox.ui.apps.models.forEach(function (app) {
            if (app.getWindow() && app.getWindow().floating) {
                app.getWindow().floating.minimize();
            }
        });
        tour.toggleBackdrop(true);
        tour.step({ labelNext: gt('Next') })
            .title(gt.format(gt.pgettext('tour', 'The %s module'), window.oxguarddata.productName))
            .content(gt.pgettext('tour', 'Security and privacy matters to everyone. That\'s why you can now send and receive encrypted emails by simply clicking on the lock icon.'))
            .waitFor('.og-ready', 10)
            .on('wait', function () {
                if (!composeApp) {
                    //compose not started, yet
                    ox.once('mail:NEW:ready, mail:new:ready', function (mode, app) {
                        composeApp = app;
                        //HACK: can not detect that mail compose window is rendered
                        composeApp.view.$el.addClass('og-ready');
                        window.setTimeout(function () {
                            composeApp.config.set('autoDismiss', true);
                            composeApp.view.securityModel.set('tour', true);
                            hotspot.add('.toggle-encryption', { 'top': 14, 'left': 14 });
                            composeApp.view.$el.closest('.io-ox-mail-compose-window').find('.toggle-encryption .fa').click();
                        }, 500);
                    });
                    ox.registry.call('mail-compose', 'open').done(function (e) {
                        composeApp = e.app;
                        composeApp.cid = 'compose.tour'; // Assign a CID for the app
                    });
                } else {
                    ox.ui.App.reuse(composeApp.cid);
                }
            })
            .end();
        tour.once('stop', function () {
            if (composeApp && composeApp.model) {
                //prevent app from asking about changed content
                composeApp.view.dirty(false);
                composeApp.quit();
            }
            def.resolve();
        });

        tour
        .step()
            //#. %s product Name
            .title(gt.format(gt.pgettext('tour', '%s Security Settings'), window.oxguarddata.productName))
            .content(resetText)
            .on('wait', function () {
                composeApp.getWindow().floating.minimize();  // Minimize the floating window
                ox.launch('io.ox/settings/main', { id: 'oxguard' }).then(function () {
                    this.setSettingsPane({ id: 'oxguard' });
                });
            })
            .waitFor('#passwordDiv')
            .referTo('#passwordDiv')
            .spotlight('#passwordDiv', { 'padding': 10 })
            .end()
        .step()
            .title((settings.get('advanced') ? gt.pgettext('tour', 'PGP Encryption Settings') : gt.pgettext('tour', 'Security Settings')))
            .content((settings.get('advanced') ?
                gt.pgettext('tour', 'From here you can update your encryption settings. For example you can choose to send encrypted emails by default.') :
                gt.pgettext('tour', 'From here, you can update your security settings.  For example, you can choose to send secure emails by default.')))
            .waitFor('.guardDefaults')
            .spotlight('.guardDefaults', { 'padding': 10 })
            .hotspot('.guardDefaults [name="defaultEncrypted"]', { 'top': 7, 'left': 7 })
            .end()
        .step()
            .title(gt.pgettext('tour', 'Signature'))
            .content(gt.pgettext('tour', 'In addition to encrypting your emails you can also sign them. This means that email recipients can be sure that the email really came from you. By ticking this checkbox your signature will be added to outgoing emails by default.'))
            .waitFor('.guardDefaults')
            .spotlight('.guardDefaults', { 'padding': 10 })
            .hotspot('.guardDefaults [name="defaultSign"]', { 'top': 7, 'left': 7 })
            .end();
        if (settings.get('advanced')) {
            tour.step()
            .title(gt.pgettext('tour', 'Alternative: PGP Inline'))
            .content(gt.format(gt.pgettext('tour', 'By default %s sends encrypted emails using PGP/Mime format. To communicate with some systems you may need to use PGP Inline.'), window.oxguarddata.productName))
            .waitFor('.guardDefaults')
            .spotlight('.guardDefaults', { 'padding': 10 })
            .hotspot('.guardDefaults [name="defaultInline"]', { 'top': 7, 'left': 7 })
            .end()
            .step()
            .title(gt.pgettext('tour', 'Advanced Settings'))
            //#. Part of the tour, showing where to manage recipient public keys
            .content(gt.pgettext('tour', 'Here you can manage PGP Public keys for your recipients.'))
            .waitFor('.scrollDone') // hack for waiting for scoll to finish
            .spotlight('.guardAdvanced [name="recipients"]', { 'padding': 10 })
            .on('wait', function () {
                $('.settings-detail-pane').animate({ scrollTop: $('.guardAdvanced [name="recipients"]').offset().top }, 400, function () {
                    $('.guardAdvanced [name="recipients"]').addClass('scrollDone');
                });
            })
            .on('show', function () {
                $('.guardAdvanced [name="recipients"]').removeClass('scrollDone');
            })
            .end()
            .step()
            .title(gt.pgettext('tour', 'Advanced Settings'))
            //#. Part of the tour, showing where you can manage your private keys
            .content(gt.pgettext('tour', 'Users can manage their private keys here.'))
            .spotlight('.guardAdvanced [name="yourKeys"]', { 'padding': 10 })
            .on('show', function () {
            })
            .end();
        } else {
            tour
            .step()
            .title(gt.pgettext('tour', 'Advanced User Settings'))
            .content($('<div>').append(
                $('<p>').text(gt.pgettext('tour', 'Enabling Advanced PGP Features will enable key management and specifying sending formats.'))
            ))
             .waitFor('[name="advanced"]')
             .hotspot('[name="advanced"]', { 'top': 7, 'left': 7 })
             .referTo($('[name="advanced"]').parent())
             .spotlight($('[name="advanced"]').parent(), { 'padding': 10 })
             .end();
        }
        tour
        .step()
            .title(gt.pgettext('tour', 'Enabling and disabling encryption'))
            .content($('<div>').append(
                $('<p>').text(gt.pgettext('tour', 'You are now able to send secure emails.')),
                $('<p>').text(gt.pgettext('tour', 'Sending without encryption is easy, too: Simply click on the lock icon again.'))
            ))
            .on('wait', function () {
                ox.ui.App.reuse(composeApp.cid);
                composeApp.getWindow().floating.toggle();

                window.setTimeout(function () {
                    hotspot.add('.toggle-encryption', { 'top': 14, 'left': 7 });
                    composeApp.view.securityModel.set('encrypt', false);
                }, 500);
            })
            .hotspot(['#statuslock'])
            .waitFor('.toggle-encryption .fa')
            .on('next', function () {
                if (composeApp && composeApp.model) {
                    //prevent app from asking about changed content
                    composeApp.view.dirty(false);
                    composeApp.quit();
                }
            })
            .end();
        if (capabilities.has('guard-drive') && capabilities.has('infostore')) {
            tour.step()
            .title(gt.pgettext('tour', 'Encrypt files'))
            .content(gt.pgettext('tour', 'You can also protect your files. Just click on Encrypt and the selected file will be stored securely.'))
            .navigateTo('io.ox/files/main')
            .waitFor('.file-list-view')
            .on('wait', function () {
                window.setTimeout(function () {
                    var firstFile = $('.file-list-view > .list-item.selectable:not(.file-type-folder):not(.file-type-guard):first');
                    if (firstFile.length > 0) {
                        firstFile.click();
                        window.setTimeout(function () {
                            $('.classic-toolbar [data-action=more] ~ ul').css('display', 'block');
                            hotspot.add('a[data-action="oxguard/encrypt"]', { 'top': 13, 'left': 10 });
                        }, 1000);
                    } else {
                        hotspot.add('.dropdown.more-dropdown', { 'top': 13, 'left': 10 });
                    }
                }, 500);
            })
            .on('next', function () {
                $('.classic-toolbar [data-action=more] ~ ul').css('display', '');
            })
            .end();
            driveText = $('<p>').append($('<a href="#">').click(openFiles).text(gt.pgettext('tour', 'Encrypt your sensitive files in Drive')));
        }
        if (location.href.indexOf('office?app') === -1) {   // We only show restarting guided tours in main tab
            tour
            .step()
                .title(gt.pgettext('tour', 'Restart Guided Tour'))
                .content(gt.pgettext('tour', '<em>Hint:</em> you can start guided tours, any time you need them, from the system menu.'))
                .hotspot('[data-name="GuardTour"]', { 'top': 13, 'left': 10 })
                .waitFor('a[data-name="GuardTour"]')
                .on('wait', function () {
                    $('#io-ox-topbar-help-dropdown-icon').find('.dropdown-toggle').click();
                })
                .on('next', function () {
                    if ($('a[data-name="GuardTour"]').css('display') === 'block') {
                        $('#io-ox-topbar-help-dropdown-icon').find('.dropdown-toggle').click();
                    }
                })
                .end();
        }
        tour
        .step()
            .title(gt.pgettext('tour', 'Guided Tour completed'))
            .content($('<div>').append(
                $('<p>').text(gt.pgettext('tour', 'You successfully learned how to protect your emails and files.')),
                $('<p>').text(gt.pgettext('tour', 'What you can do now:')),
                $('<p>').append($('<a href="#">').click(function () {
                    writeMail();
                    tour.close();
                }).text(gt.pgettext('tour', 'Write an encrypted email'))),
                driveText,
                $('<hr>'),
                $('<p>').append($('<a href="#">').click(setEncryptDefault).text(gt.pgettext('tour', 'Activate encryption for all new emails'))),
                $('<p>').append($('<a href="#">').click(openSettings).text(gt.pgettext('tour', 'Review your encryption settings'))),
                $('<p class="help">').append(new HelpView({
                    base: 'help',
                    content: gt.pgettext('tour', 'Learn more')
                }).render().$el)
            ))
            .end()
        .start();
        return def;
    });

    Tour.registry.add({
        id: 'default/oxguard/createKeys',
        priority: 1
    }, function () {
        var def = $.Deferred();

        var configured = util.isGuardConfigured();

        if (_.device('small') || _.device('smartphone')) {
            if (configured) return;  // no tour for mobile
            return keys.createKeys();
        }

        var tour = new Tour(),
            model = new keys.CreateKeysViewModel({
                initialSetup: true
            }),
            view = new keys.CreateKeysView({
                model: model
            });

        tour.once('stop', function () {
            if (!model.get('pending')) { // awaiting some other promise
                if (model.get('sent') === false) {
                    def.reject('cancel');
                } else {
                    def.resolve('OK');
                }
            }

            view.remove();
            model.off();
            view = model = null;
        });
        //Guard is not configured, yet.
        if (!configured) {
            tour.step({ labelNext: gt('Start Setup') })
            .title(gt.format(gt.pgettext('tour', 'Setup %s'), window.oxguarddata.productName))
            .content($('<div>').append(
                //#. %s product name
                $('<p>').text(gt.format(gt.pgettext('tour', 'Welcome to %1$s. You will now be taken on a tour, showing you how to use %1$s.'),
                    window.oxguarddata.productName)))
                .append($('<p>').text(gt.format(gt.pgettext('tour', 'Firstly however, you’ll need to setup %1$s by choosing a password which will be used to encrypt or decrypt items such as email or files.'
                ), window.oxguarddata.productName)))
            )
            .end()
        .step()
            //#. %s product name
            .title(gt('Choose %s Password', window.oxguarddata.productName))
            .content(view.render().$el)
            .beforeShow(function () {
                var self = this;
                this.toggleNext(false);
                model.on('change', function (model) {
                    self.toggleNext(model.isValid());
                });
                model.once('send:ok', function () {
                    //HACK: no way to pause the wizard until some event
                    $('body').addClass('og-ready');
                });
                window.setTimeout(function () {
                    if (_.device('desktop')) {
                        $('#newogpassword').focus();
                    } else {
                        $('[type="password"]').removeAttr('readonly');  // Remove the read only but do not focus.  User needs to click for keyboard
                    }
                }, 1000);
            })
            .on('next', function () {
                model.send();
            })
            .end();
        }

        // If we are already configured, can start with the offer for more tour
        if (configured) {
            $('body').addClass('og-ready');
        }

        tour
        .step({ labelDone: gt('Close') })
            //#. %s product Name
            .title(gt.format(gt.pgettext('tour', '%s set up completed'), window.oxguarddata.productName))
            .content($('<div>').append(
                $('<p>').text(gt.format(gt.pgettext('tour', 'Congratulations, you have successfully setup %s'), window.oxguarddata.productName)),
                $('<p>').text(gt.format(gt.pgettext('tour', 'Now you can learn more about how to use %s by following the links below:'), window.oxguarddata.productName)),
                $('<p>').append($('<a id="startTourLink" href="#">').click(function () {
                    model.set('pending', true);
                    $('.wizard-step .btn-primary').click();
                    Tour.registry.get('default/oxguard').get('run')().then(def.resolve, def.resolve);
                    //#. %s product name
                }).text(gt('Guided tour for %s', window.oxguarddata.productName))),
                $('<p>').append(new HelpView({
                    base: 'help',
                    content: gt.format(gt.pgettext('tour', 'Learn more about %s'), window.oxguarddata.productName)
                }).render().$el)
            ))
            .mandatory()
            .beforeShow(function () {
                //no going back, from here
                this.toggleBack(false);
            })
            .on('ready', function () {
                //HACK: no way to pause the wizard until some event
                $('body').removeClass('og-ready');
            })
            .waitFor('.og-ready')
            .end()
        .start();

        return def;
    });

    return Tour.registry;
});
