/**
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016-2020 OX Software GmbH
 */

define('oxguard/register', [
    'io.ox/core/extensions',
    'gettext!oxguard',
    'settings!oxguard',
    'io.ox/core/capabilities',
    'less!oxguard/style'
], function (ext, gt, settings, capabilities) {

    'use strict';

    var productName = ox.serverConfig['guard.productName'] === undefined ? settings.get('productName') : ox.serverConfig['guard.productName'];
    if (productName === undefined || productName === 'undefined' || productName === '') {
        productName = 'Guard';
    }

    var oxguarddata = {
        passcode: null,
        settings:  {},
        productName: sanitize(productName)
    };

    window.oxguarddata = oxguarddata;

    if (settings.get('pgplocal') === true) {  // Load local keys if set
        require(['pgp_local/register'], function (pgpLocal) {
            pgpLocal.loadPrivate();
        });
    }

    // Load data generated at compile (build version, etc)
    require(['oxguard/oxguard_data'], function (generatedData) {
        _.extend(window.oxguarddata, generatedData);
    });

    // Login section
    sendauth(''); // initial login, no password yet

    // Default settings if needed
    if (settings.get('advanced') === undefined) {
        settings.set('advanced', !!settings.get('defaultAdvanced'));
        settings.save();
    }

    // Sends first login to get settings info, key data
    function sendauth(pass) {
        require(['oxguard/oxguard_core'], function (oxguard) {
            oxguard.auth(ox.user_id, pass)
            .done(function (data) {
                try {
                    if (data.error) {
                        console.error(data.error);
                        console.debug('Unable to connect to Encryption Server');
                        return;
                    }
                    oxguarddata = _.extend(window.oxguarddata, checkLanguages(data));
                    oxguarddata.loaded = true;
                    if (data.auth.length > 20 && !(/^Error:/i.test(data.auth))) { // If good code comes back, then store userid
                        oxguarddata.userid = ox.user_id;
                        oxguarddata.passcode = data.auth;
                    } else { // If bad code, then log and mark oxguard.user_id as -1
                        if (/^((Error)|(Bad)|(Lockout))/i.test(data.auth)) {
                            oxguarddata.passcode = null;
                        } else {
                            oxguarddata.passcode = data.auth;
                        }
                        if (data.error) {
                            require(['io.ox/core/notifications'], function (notify) {
                                notify.yell('error', productName + '\r\n' + data.error);
                            });
                        }
                        oxguarddata.userid = -1;
                    }
                    if (settings.get('defaultEncrypted') === undefined) {
                        require(['oxguard/settings/pane'], function (defaults) {
                            defaults.initSettings(data.settings);
                        });
                    }
                    if (ox.context_id === 0) ox.context_id = data.cid; // If the cid wasn't loaded, update from backend
                    oxguarddata.baton = window.oxguarddata.baton;
                    oxguarddata.recovery = data.recoveryAvail === undefined ? !data.noRecovery : data.recoveryAvail;
                    window.oxguarddata = oxguarddata;
                    // Check for any pending onload actions
                    if (oxguarddata.onload) {
                        try {
                            oxguarddata.onload.forEach(function (load) {
                                load.call();
                            });
                            oxguarddata.onload = null;
                        } catch (e) {
                            console.error('Problem executing onload functions for Guard ' + e);
                        }
                    }
                    if (sessionStorage !== null) { // If we were passed login info from login screen
                        try {
                            if (sessionStorage.oxguard && (sessionStorage.oxguard !== 'null')) {
                                var p = sessionStorage.oxguard;
                                sessionStorage.oxguard = null;
                                sendauth(p);
                            }
                        } catch (e) {
                            console.log('Private mode');
                            return;
                        }
                    }
                } catch (e) {
                    console.log(e);
                    console.debug('Unable to connect to the encryption server');
                }
            })
            .fail(function () {
                console.debug('Unable to connect to the encryption server');
            });
        });

    }

    // Parse out available languages, comparing Guard templates and UI available
    function checkLanguages(data) {
        var guardLanguages = data.lang;
        var available = {};
        for (var lang in guardLanguages) {
            if (ox.serverConfig.languages[lang]) {
                available[lang] = guardLanguages[lang];
            }
        }
        data.lang = available;
        return data;
    }

    function doLogout(notify) {
        var def = $.Deferred();
        window.oxguarddata.passcode = null;
        require(['oxguard/oxguard_core'], function (oxguard) {
            oxguard.saveAuthToSession('', -1)  // destroy auth token
            .done(function () { // Destroy Guard mapping and auth-token
                if (notify) {
                    require(['io.ox/core/notifications'], function (notify) {
                        notify.yell('success', gt('Logged out of %s', productName));
                    });
                }
            });
        });

        return def;
    }

    ext.point('io.ox/core/appcontrol/right/help').extend({
        id: 'guard-guided-tour',
        index: 290,
        extend: function () {
            if (_.device('small')) return;
            if (capabilities.has('guard-mail') || capabilities.has('guard-drive')) {
                this.append(
                    this.link('GuardTour', gt('Guided tour for %s', window.oxguarddata.productName), function (e) {
                        e.preventDefault();
                        require(['io.ox/core/tk/wizard', 'oxguard/tour/main'], function (Tour) {
                            Tour.registry.run('default/oxguard');
                        });
                    })
                );
            }
        }
    });

    ext.point('io.ox/core/appcontrol/right/account/signouts').extend({
        id: 'logoutOG',
        index: 50,
        extend: function () {
            this.link('logoutOG', gt('Sign out %s', window.oxguarddata.productName), function (e) {
                e.preventDefault();
                window.oxguarddata.passcode = null;
                doLogout(true);
            });
            this.$el.on('shown.bs.dropdown', function () {
                var pass = window.oxguarddata.passcode;
                if (pass && pass.length > 10) {
                    $('[data-name="logoutOG"]').show();
                } else {
                    $('[data-name="logoutOG"]').hide();
                }
            });
        }
    });

    function sanitize(data) {
        var regex = new RegExp('<(?:!--(?:(?:-*[^->])*--+|-?)|script\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</script\\s*|style\\b(?:[^"\'>]|"[^"]*"|\'[^\']*\')*>[\\s\\S]*?</style\\s*|/?[a-z](?:[^"\'>]|"[^"]*"|\'[^\']*\')*)>', 'gi');
        return (data.replace(regex, ''));
    }

});
