define('oxguard/mail/register_compose', [
    'io.ox/core/extensions',
    'gettext!oxguard',
    'io.ox/backbone/mini-views/dropdown',
    'oxguard/mail/oxguard_mail_compose_core',
    'oxguard/util',
    'pgp_mail/toggle-encryption',
    'io.ox/backbone/mini-views/helplink',
    'oxguard/mail/keymanager',
    'settings!oxguard',
    'oxguard/mail/security_model',
    'io.ox/mail/compose/api',
    'less!oxguard/mail/style'
], function (ext, gt, Dropdown, core, util, ToggleEncryption, HelpView, keyman, settings, SecurityModel, composeAPI) {

    'use strict';

    // Mobile only

    ext.point('io.ox/mail/compose/fields').extend({
        id: 'lock-mobile',
        index: 250,
        draw: function (baton) {
            if (_.device('!smartphone')) return;
            if (!util.hasCryptoCapability()) return;
            var node = this;
            var securityModel = new SecurityModel(baton.model.get('security'), baton.model);
            baton.view.securityModel = securityModel;
            var view = new ToggleEncryption.ViewMobile({
                model: securityModel
            }, baton.model);
            view.noLinkMail(baton.view);
            baton.view.toggleEncryption = view;
            node.find('.sender').append(view.render().$el);
            node.find('.mail-input').addClass('guardMail');
            setupCompose(baton);
        }
    });

    ext.point('io.ox/mail/compose/menu-mobile').replace({
        id: 'security',
        index: 100,
        draw: function (baton) {
            var node = this;
            if (_.device('!smartphone')) return;
            drawDropDownMobile(baton, $(node));
        }
    });

    // Desktop only
    ext.point('io.ox/mail/compose/fields').extend({
        id: 'info-and-options',
        index: 900,
        draw: function (baton) {
            if (_.device('smartphone')) return;
            if (!util.hasCryptoCapability()) return;

            // row
            var composeUtil = require('io.ox/mail/compose/util'),
                node = composeUtil.infoLine({
                    icon: 'fa-lock',
                    text: gt('This email will be encrypted.')
                }).addClass('stripes-green encrypted');
            this.append(node);

            // toggle
            update();
            baton.view.securityModel.on('change:encrypt', update);
            function update() {
                node.toggleClass('hidden', !baton.view.securityModel.get('encrypt'));
            }

            // dropdown
            if (!window.oxguarddata || !window.oxguarddata.settings || !window.oxguarddata.settings.oxguard) return;
            var dropdown = new Dropdown({
                model: baton.view.securityModel,
                label: gt('Options'),
                caret: true
            })
            .header(gt('Security'))
            .option('PGPSignature', true, gt('Sign email'))
            .option('PubKey', true, gt('Attach my public key'));

            if (settings.get('advanced')) {
                dropdown
                //#. Format of the email, HTML or Plaintext
                .header(gt('PGP Format'))
                .option('PGPFormat', 'mime', gt('Mime'))
                .option('PGPFormat', 'inline', gt('Inline'));
            }

            //  Help icon
            dropdown.render().$el
                .addClass('security-options')
                .find('ul').prepend(
                    new HelpView({
                        base: 'help',
                        iconClass: 'guardhelp fa-question-circle fa',
                        href: 'ox.appsuite.user.sect.guard.email.send.html',
                        tabindex: '-1'
                    }).render().$el.addClass('guard-help-icon')
                );

            node.append(dropdown.$el);
        }
    });

    ext.point('io.ox/mail/compose/composetoolbar').extend({
        id: 'lock',
        index: 400,
        draw: function (baton) {
            if (_.device('smartphone')) return;
            // TODO check for encr reply
            if (!util.hasCryptoCapability()) return;

            var container = $('<li role="presentation">');
            var securityModel = new SecurityModel(baton.model.get('security'), baton.model),
                view = new ToggleEncryption.View({
                    tagName: 'a',
                    model: securityModel,
                    mailModel: baton.model
                });

            baton.view.securityModel = securityModel;
            baton.view.toggleEncryption = view;
            view.noLinkMail(baton.view);

            this.append(
                container.append(view.render().$el)
            );

            setupCompose(baton);
        }
    });

    ext.point('io.ox/mail/compose/menuoptions').extend({
        id: 'guard-options',
        index: 400,
        draw: function (baton) {
            if (_.device('smartphone')) return;
            if (!util.hasCryptoCapability()) return;

            var intermediateModel = this.data('view').model,
                securityModel = baton.view.securityModel;

            // sync updates of intermediateModel (dropdown) and securityModel
            ['PGPSignature', 'PubKey'].forEach(function (key) {
                intermediateModel.set(key, securityModel.get(key));
                intermediateModel.on('change:' + key, function name() {
                    securityModel.set(key, intermediateModel.get(key));
                });
                securityModel.on('change:' + key, function name() {
                    intermediateModel.set(key, securityModel.get(key));
                });
            });

            this.data('view')
                .divider()
                .header(gt('Security'))
                .option('PGPSignature', true, gt('Sign email'))
                .option('PubKey', true, gt('Attach my public key'));
        }
    });

    function setupCompose(baton) {
        // Check the options for the page. Lock icon settings, etc
        require(['oxguard/mail/options_new'], function (options) {
            options.createOptions(baton);
        });
    }

    // Check if attachment is PGP file, and set email to encrypt if found
    function checkEncrAttachmentAdded(val, baton) {
        if ((val.get('file_mimetype') === 'application/pgp-enrypted') ||
                (val.get('filename') && val.get('filename').indexOf('.pgp') > 0)) {
            require(['io.ox/core/capabilities'], function (capabilities) {
                if (capabilities.has('guard-mail')) {
                    baton.view.securityModel.set('encrypt', true);
                }
            });
        }
    }

    // Add to attachment extensions monitor to see if attaching PGP file
    ext.point('io.ox/mail/compose/attachments').extend({
        id: 'checkPGPAttachment',
        index: 200,
        draw: function (baton) {
            baton.model.get('attachments').on('add', function (val) {
                checkEncrAttachmentAdded(val, baton);
            });
        }
    });

    // Token and key handling

    ext.point('io.ox/mail/compose/createtoken').extend({
        id: 'guardToken',
        action: function (baton) {
            if (baton.view.securityModel.get('encrypt') === true) {
                var target = $(baton.event.relatedTarget);
                var email = baton.event.attrs.model.get('token').value;
                // OK, check the keys
                keyman.checkRecips(email, baton)
                .done(function (keyResults) {
                    core.drawKeyIcons(keyResults[email], target);
                });
                // Handle tooltip with keyboard nav
                target.on('focus', function () {
                    if (target.hasClass('grabbed')) {
                        // Don't show if grabbed
                        target.find('.key_tooltip').tooltip('hide');
                    } else {
                        target.find('.key_tooltip').tooltip('show');
                    }
                });
                target.on('blur', function () {
                    target.find('.key_tooltip').tooltip('hide');
                });
                // If moving token around, hide tooltip
                target.on('keyup', function (e) {
                    if (e.which === 32) {
                        target.find('.key_tooltip').tooltip('hide');
                    }
                });
            }
        }
    });

    // If token removed, then remove it from our list of gkeys
    ext.point('io.ox/mail/compose/removetoken').extend({
        id: 'guardDelToken',
        action: function (baton) {
            if (baton.config.get('gkeys')) {
                var email = baton.event.attrs.model.get('token').value;
                require(['oxguard/mail/keymanager'], function (keyman) {
                    keyman.deleteRecip(email, baton);
                });
            }
        }
    });

    // Send extension points

    ext.point('io.ox/mail/compose/actions/send').extend(
        {
            // Check if all keys have been checked
            id: 'keysDoneCheck',
            index: 100,
            before: 'busy:start',
            perform: function (baton) {
                if (!baton.view.securityModel.get('encrypt')) return true;
                var def = $.Deferred();
                if (isReady(baton)) {
                    def.resolve();
                } else {
                    waitUntilReady(baton, def);
                }
                return (def);
            }
        },
        {
            // Check if password needed for signing
            id: 'signCheck',
            index: 200,
            before: 'busy:start',
            perform: function (baton) {
                var def = $.Deferred();
                var security = baton.model.get('security');
                if (security.sign) {
                    require(['oxguard/auth'], function (auth) {
                        var options = {
                            optPrompt: gt('Please enter your %s password to sign this email.', window.oxguarddata.productName),
                            minSingleUse: true
                        };
                        auth.authorize(baton, options)
                        .done(function (auth) {
                            var security = baton.model.get('security');
                            security.authentication = auth;
                            baton.model.set('security', security);
                            if (baton.model.get('security').encrypt) {
                                checkKeys(baton, def);
                            } else {
                                def.resolve();
                            }
                        })
                        .fail(function () {
                            baton.stopPropagation();
                            def.reject();
                        });
                    });
                } else if (security.encrypt) {
                    checkKeys(baton, def);
                } else {
                    def.resolve();
                }
                return (def);
            }
        },
        {
            // Check if encrypted files attached that need password for decryption
            id: 'encryptedFiles',
            index: 300,
            before: 'busy:start',
            perform: function (baton) {
                if (!baton.view.securityModel.get('encrypt')) return true;
                var security = baton.model.get('security');
                var def = $.Deferred();
                var attachments = baton.model.get('attachments');
                var found = false;
                attachments.models.forEach(function (attach) {
                    if (found) return;  // Only prompt once
                    var mimeType = attach.get('file_mimetype');
                    var fileName = attach.get('filename');
                    if ((mimeType && (mimeType.indexOf('pgp-encrypted') > -1)) || (fileName && (fileName.indexOf('.pgp') > -1))) {
                        found = true;
                        require(['oxguard/auth'], function (auth) {
                            var options = {
                                optPrompt: gt('There appears to be an encrypted attachment.  Please enter your %s password to decode the attachment for sending.', window.oxguarddata.productName),
                                minSingleUse: true
                            };
                            auth.authorize(baton, options)
                            .done(function (auth) {
                                security.authentication = auth;
                                baton.model.set('security', security);
                                def.resolve();
                            })
                            .fail(function () {
                                def.reject();
                            });
                        });
                    }
                });
                if (!found) def.resolve();
                return (def);
            }
        },
        {
            // Check for authentication errors
            id: 'checkAuthErrors',
            index: 2900,
            after: 'send',
            perform: function (baton) {
                if (baton.error && containsError(baton.errorCode, baton.error)) {
                    handleFail(baton);
                    handleReauth(gt('Please re-enter your %s password before sending.', window.oxguarddata.productName), baton)
                    .done(function () {
                        updateAuth(baton.model).then(function () {
                            baton.view.send();
                        });
                    });
                    return new $.Deferred().reject();
                }
            }
        }
    );

    function updateAuth(model) {
        var security = model.get('security');
        security.authToken = security.authentication;
        delete security.authentication;
        return composeAPI.space.update(model.get('id'), { security: security });
    }

    // Save extension points

    ext.point('io.ox/mail/compose/actions/save').extend({
        id: 'checkAuthErrors',
        index: 1050,
        perform: function (baton) {
            if (baton.error && containsError(baton.errorCode, baton.error)) {
                handleFail(baton);
                handleReauth(gt('Please re-enter your %s password before saving.', window.oxguarddata.productName), baton)
                .done(function () {
                    updateAuth(baton.model);
                });
                baton.stopPropagation();
                return new $.Deferred().reject();
            }
        }
    });

    // GET response event listener
    ext.point('io.ox/mail/compose/actions/get/error').extend({
        index: 100,
        id: 'guardAuthError',
        handler: function (baton) {
            if (baton.result && containsError(baton.result.code, baton.result.error)) {
                baton.handled = $.Deferred();
                return handleReauth(gt('Please enter your %s password to edit this encrypted draft.', window.oxguarddata.productName), baton)
                .then(function () {
                    // TODO: this needs proper converters and serializers/deserializers for compose model <=> guard model <=> JSON
                    var security = baton.model.get('security');
                    security.authToken = security.authentication;
                    delete security.authentication;
                    return composeAPI.space.update(baton.model.get('id'), { security: security });
                }).then(function () {
                    baton.handled.resolve(composeAPI.space.get(baton.model.get('id')));
                }).catch(function (error) {
                    baton.handled.reject(error);
                });
            }
        }
    });

    var reauthorizing = false;

    ext.point('io.ox/mail/compose/boot').extend({
        index: 'last',
        id: 'guardAuthSaveError',
        perform: function (baton) {
            baton.model.on('fail:save', function (e) {
                if (e.error && e.code) {
                    if (containsError(e.code, e.error)) {
                        var model = this;
                        handleReauth(gt('Please re-enter your %s password before saving.', window.oxguarddata.productName), baton)
                        .done(function () {
                            updateAuth(model).then(function () {
                                model.save();
                            });
                        });
                    }
                }
            });
        }
    });

    ext.point('io.ox/mail/compose/autosave/error').extend({
        index: 1,
        id: 'guardAuthError',
        handler: function (baton) {
            if (reauthorizing) return;
            if (containsError(baton.code, baton.error)) {
                baton.preventDefault();
                if (reauthorizing) return;
                reauthorizing = true;
                var security = baton.model.get('security');
                require(['oxguard/auth'], function (auth) {
                    var options = {
                        optPrompt: gt('Please re-enter your %s password.', window.oxguarddata.productName),
                        minSingleUse: true,
                        forceRelogin: true
                    };
                    auth.authorize(baton, options)
                    .done(function (auth) {
                        if (security) {  // Update security authorization
                            security.authentication = auth;
                            baton.view.model.set('security', security);
                        }
                        baton.view.autoSaveDraft();  // Try again
                    })
                    .always(function () {
                        reauthorizing = false;
                        baton.returnValue.reject(); // Reject the action as needed re-auth
                    });
                });

            }
        }
    });

    // Functions

    // Drawing dropdown options for security
    function drawDropDownMobile(baton, node) {
        if (window.oxguarddata && window.oxguarddata.settings) {
            if (window.oxguarddata.settings.oxguard) {
                var dropdown = new Dropdown({ model: baton.view.securityModel, label: gt('Security'), caret: true });
                if (settings.get('advanced')) {
                    dropdown
                        .header(gt('PGP'));
                } else {
                    dropdown
                    .header(gt('Security'));
                }
                dropdown
                    .option('encrypt', true, (settings.get('advanced') ? gt('Encrypt') : gt('Secure')))
                    .option('PGPSignature', true, gt('Sign'));
                if (settings.get('advanced')) {
                    dropdown
                    //#. Format of the email, HTML or Plaintext
                    .header(gt('Format'))
                    .option('PGPFormat', 'mime', gt('PGP Mime'))
                    .option('PGPFormat', 'inline', gt('PGP Inline'));
                    dropdown
                    .header(gt('Keys'))
                    .option('PubKey', true, gt('Attach my key'));
                }
                node.addClass('security-mobile')
                    .append(dropdown.render().$el.addClass('text-left'));
            }
        }
    }

    // Check if any of the recipients are Guest and prompt for greeting/password
    function checkKeys(baton, def) {
        var keys = baton.config.get('gkeys');
        var isGuest = false;
        var fail = false;
        for (var email in keys) {
            if (keys[email].result === 'guest') isGuest = true;
            if (keys[email].result === 'fail') fail = true;
        }
        if (fail) {
            require(['io.ox/core/notifications'], function (notifications) {
                notifications.yell('error', gt('Unable to find keys for all recipients.'));
            });
            baton.stopPropagation();
            def.reject();
            return;
        }
        if (!isGuest) {
            def.resolve();
            return;
        }
        require(['oxguard/mail/guestPrompt'], function (prompt) {
            prompt.guestOptions(baton)
            .done(function (data) {
                var security = _.extend(baton.model.get('security'), data);
                baton.model.set('security', security);
                def.resolve();
            })
            .fail(function () {
                baton.stopPropagation();
                def.reject();
            });
        });
    }

    // Check all keys ready
    function isReady(baton) {
        var keys = baton.config.get('gkeys');
        if (keys === undefined) return true; // No keys
        var pending = false;
        for (var email in keys) {
            pending = pending || keys[email].pending;
        }
        return !pending;
    }

    // Check if response code and error string contain auth error
    function containsError(code, string) {
        if (code === 'GRD-MW-0001') return true;  // Midleware Auth error
        if (code !== 'GUARD-0003') return false;
        return (string && string !== null &&
            (string.indexOf('GRD-AUTH-0002') > -1 || string.indexOf('GRD-PGP-0005') > -1));
    }

    function handleFail(baton) {
        if (baton.error && !baton.warning) {
            baton.stopPropagation();
            var win = baton.app.getWindow();
            if (win) {
                // reenable the close button in toolbar
                if (baton.closelink) {
                    baton.closelink.show();
                }
                win.idle().show();
            }
            baton.app.launch();
            return;
        }
        return;
    }

    function handleReauth(prompt, baton) {
        window.oxguarddata.passcode = null;  // wipe stored auth token
        var def = $.Deferred();
        require(['oxguard/auth'], function (auth) {
            var security = baton.model.get('security');
            var options = {
                optPrompt: prompt,
                minSingleUse: true,
                forceRelogin: true
            };
            auth.authorize(baton, options)  // authorize and force relogin
            .done(function (auth) {
                security.authentication = auth;
                baton.model.set('security', security);
                baton.stopPropagation();
                def.resolve();
            })
            .fail(function (data, e) {
                // return a ox error object (fallback build one)
                var error = _.extend({ code: 'unknown', error: gt('An error occurred. Please try again.') }, e || {});

                // adjust error message to be less technical
                if (data === 'cancel') _.extend(error, { code: 'GRD-UI-0001', error: gt('Authorization failed as password prompt was closed by user') });
                if (error.code === 'GRD-MW-0003') error.message = gt('Bad password');
                if (error.code === 'GRD-MW-0005') error.message = gt('Account Locked out');

                // flag error objects that (temporary) breaks access to composition spaces
                if (/^(GRD-UI-0001|GRD-MW-0003|GRD-MW-0005)$/.test(error.code)) error.critical = true;

                baton.stopPropagation();
                def.reject(error);
            });
        });
        return def;
    }

    // If not all keys returned, wait until completed.
    // Timeout if no backend response
    function waitUntilReady(baton, def) {
        baton.waiting = true;
        var win = baton.view.app.getWindow();
        win.busy();
        baton.model.on('change:gkeys', function () {
            if (isReady(baton)) {
                win.idle();
                def.resolve();
                baton.waiting = false;
            }
        });
        window.setTimeout(function () {
            if (baton !== undefined) {
                if (baton.waiting === true) {
                    win.idle();
                    baton.waiting = false;
                    def.resolve();
                }
            }
        }, 15000);
    }

});
