/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH. group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.guard.backend.mailfilter.sieve;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.List;
import java.util.Set;
import org.apache.jsieve.SieveException;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.capabilities.CapabilityService;
import com.openexchange.config.lean.LeanConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.jsieve.commands.ActionCommand;
import com.openexchange.mail.filter.json.v2.json.fields.GeneralField;
import com.openexchange.mail.filter.json.v2.json.mapper.parser.CommandParserJSONUtil;
import com.openexchange.mail.filter.json.v2.json.mapper.parser.action.external.FilterActionRegistry;
import com.openexchange.mail.filter.json.v2.json.mapper.parser.action.external.SieveFilterAction;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link GuardEncryptAction}
 *
 * @author <a href="mailto:greg.hill@open-xchange.com">Greg Hill</a>
 * @since v7.10.4
 */
public class GuardEncryptAction implements SieveFilterAction {

    /**
     * Required Guard capability
     */
    private final static String GUARD_CAPABILITY = "guard-mail";
    private final ServiceLookup services;
    /*
     * Name of this Action
     */
    private static final String NAME = "guard_encrypt";

    public GuardEncryptAction(ServiceLookup services) {
        this.services = services;
    }

    @Override
    public String getJsonName() {
        return NAME;
    }

    /**
     * Gets the {@link FilterActioNRegistry}
     *
     * @return
     * @throws OXException
     */
    private FilterActionRegistry getRegistry() throws OXException {
        return services.getServiceSafe(FilterActionRegistry.class);
    }

    /**
     * Return the LeanConfigurationService
     *
     * @return The {@link LeanConfigurationService}
     * @throws OXException
     */
    private LeanConfigurationService getConfig() throws OXException {
        return services.getServiceSafe(LeanConfigurationService.class);
    }

    /**
     * Obtains the defined script from configuration
     *
     * @return The script name
     * @throws OXException
     */
    private String getScript() throws OXException {
        return getConfig().getProperty(SieveConfig.guardEncryptScript);
    }

    /**
     * Checks whether this action is enabled or not
     *
     * @return <code>true</code> if it is enabled, <code>false</code> otherwise
     * @throws OXException
     */
    private boolean isEnabled() throws OXException {
        return getConfig().getBooleanProperty(SieveConfig.sieveEnabled);
    }

    @Override
    public boolean isCommandSupported(Set<String> capabilities) throws OXException {
        if (capabilities.contains("vnd.dovecot.filter")) {
            return isEnabled();
        }
        return false;
    }

    @SuppressWarnings("unchecked")
    @Override
    public boolean isApplicable(ActionCommand actionCommand) throws OXException {
        final ArrayList<Object> arguments = actionCommand.getArguments();
        // Confirm that this Filter command refers to the encrypt action
        return (!arguments.isEmpty() && arguments.get(0) instanceof ArrayList && ((ArrayList<String>) arguments.get(0)).contains(getScript()));
    }

    @Override
    public void parse(JSONObject jsonObject, ActionCommand actionCommand) throws OXException {
        try {
            jsonObject.put(GeneralField.id.name(), NAME);
        } catch (JSONException e) {
            throw OXException.general("Problem creating mailfilter rule", e);
        }
    }

    @Override
    public boolean isApplicable(JSONObject jsonObject) throws OXException {
        try {
            return (jsonObject.has("id") && jsonObject.get("id").equals(NAME));
        } catch (JSONException e) {
            throw OXException.general("Error testing Encrypt Action", e);
        }
    }

    @Override
    public ActionCommand parse(JSONObject jsonObject, ServerSession session) throws OXException {
        // Verify permission
        if (services.getServiceSafe(CapabilityService.class).getCapabilities(session.getUserId(), session.getContextId()).contains(GUARD_CAPABILITY)) {
            try {
                // Add required arguments, name of the script, and the user Id and context
                final ArrayList<Object> arrayList = new ArrayList<Object>();
                arrayList.add(CommandParserJSONUtil.stringToList(getScript()));
                final List<String> args = new ArrayList<String>(2);
                args.add(Integer.toString(session.getUserId()));
                args.add(Integer.toString(session.getContextId()));
                arrayList.add(args);
                return new ActionCommand(getRegistry(), arrayList);
            } catch (SieveException ex) {
                throw OXException.general("Problem executing encryption sieve filter", ex);
            }
        }
        throw OXException.noPermissionForModule("guard");  // Doesn't have the capability to perform this action
    }

    @Override
    public Hashtable<String, Integer> getTagArgs() {
        return new Hashtable<String, Integer>();
    }

}
